import * as React from "react";

import gql from "graphql-tag";
import { action, observable } from "mobx";
import { observer } from "mobx-react";
import { Mutation } from "react-apollo";

import { WithConfirmationModal } from "aegis/functionality/components/confirmation-modal";
import { DeleteVodAndClipsButtonComponent } from "./component";

export interface Props {
  id: string;
  forceDisabledReason?: string;
}

export type DeleteVodAndClipsResponse = {
  deleteVodAndClips: {
    success: boolean;
  };
};

export type DeleteVodAndClipsInput = {
  id: string;
};

export const DELETE_VOD_AND_CLIPS = gql`
  mutation deleteVod($id: ID!) {
    deleteVodAndClips(input: { id: $id }) {
      success
    }
  }
`;

export class DeleteVodAndClipsMutation extends Mutation<DeleteVodAndClipsResponse, DeleteVodAndClipsInput> {}

@observer
export class DeleteVodAndClipsButton extends React.Component<Props> {
  @observable private deleted: boolean = false;
  @observable private error: Error | undefined = undefined;

  public render() {
    const { forceDisabledReason, id } = this.props;
    const error = this.error;
    const deleted = this.deleted;

    return (
      <DeleteVodAndClipsMutation mutation={DELETE_VOD_AND_CLIPS} onError={this.onError} onCompleted={this.onCompleted}>
        {(deleteVodAndClips, { loading }) => {
          return (
            <WithConfirmationModal
              title="Delete VOD and Clips"
              body="Are you sure you want to delete this VOD and all Clips created from it?"
              confirmLabelOverride="Delete"
              onConfirm={() => {
                deleteVodAndClips({ variables: { id } });
              }}
            >
              {confirm => {
                return (
                  <DeleteVodAndClipsButtonComponent
                    onClick={confirm}
                    error={error}
                    forceDisabledReason={forceDisabledReason}
                    deleted={deleted}
                    loading={loading}
                  />
                );
              }}
            </WithConfirmationModal>
          );
        }}
      </DeleteVodAndClipsMutation>
    );
  }

  @action
  private onError = (error: Error) => {
    console.error("Failed to delete chatroom: ", error);
    this.error = error;
  };

  @action
  private onCompleted = (response: DeleteVodAndClipsResponse) => {
    this.deleted = response.deleteVodAndClips.success;
  };
}
