import * as React from "react";

import gql from "graphql-tag";

import { Report } from "aegis/features/reports";
import { ReportSubtypeFields } from "aegis/fragments";
import { Query } from "aegis/functionality/apollo-wrapper";
import { Color, CoreText, Placeholder } from "twitch-core-ui";
import { ReportEmbedComponent } from "./component";

export interface Props {
  report: Report;
}

export type ReportSubtypeFieldsInput = {
  id: string;
};

export type ReportSubtypeFieldsResponse = {
  report: Report;
};

export const FETCH_REPORT_SUBTYPE_FIELDS = gql`
  query reportSubtypeFields($id: ID!) {
    report(id: $id) {
      ...ReportSubtypeFields
    }
  }
  ${ReportSubtypeFields}
`;

export class ReportEmbedQuery extends Query<ReportSubtypeFieldsResponse, ReportSubtypeFieldsInput> {}

export class ReportEmbed extends React.Component<Props> {
  public render() {
    const { report } = this.props;

    return (
      <ReportEmbedQuery
        displayName="ReportEmbed"
        fallbackComponent={<ReportEmbedFallback report={report} />}
        query={FETCH_REPORT_SUBTYPE_FIELDS}
        variables={{ id: report.id }}
      >
        {({ loading, error, data }) => {
          if (loading) {
            return <Placeholder lineCount={5} />;
          }

          if (error) {
            console.error("An error occurred ", error);
            return (
              <>
                <CoreText color={Color.Error}>
                  An error occurred fetching report details. Error: {error.message}.
                </CoreText>
                <ReportEmbedFallback report={report} />
              </>
            );
          }

          return <ReportEmbedComponent report={{ ...report, ...data!.report }} />;
        }}
      </ReportEmbedQuery>
    );
  }
}

const ReportEmbedFallback: React.SFC<Props> = ({ report }) => {
  return <div>Raw Context: {report && report.rawContext}</div>;
};
