import * as React from "react";

import { UserReport } from "aegis/features/reports";
import { User } from "aegis/features/user";
import { VodVideoEmbed } from "aegis/features/vod";
import { VodPopoutButton } from "aegis/features/vod-popout-button";
import { IndicatorButton } from "aegis/functionality/components/indicator-button/component";
import { action, observable } from "mobx";
import { observer } from "mobx-react";
import {
  Button,
  ButtonSize,
  ButtonType,
  Color,
  CoreText,
  Display,
  FlexDirection,
  FormGroup,
  JustifyContent,
  Layout,
  StyledLayout,
  SVGAsset,
  TextAlign
} from "twitch-core-ui";

export interface Props {
  report: UserReport;
}

export enum TestSelectors {
  VOD_EMBED = "user-embed--vod-embed",
  VOD_EMBED_TOGGLE_BUTTON = "user-embed--vod-embed-toggle-button"
}

@observer
export class UserEmbed extends React.Component<Props> {
  @observable private showVodEmbed: boolean = false;

  public render() {
    const { similarClipsURL, squad, vod, chatPopupURL, createdAt } = this.props.report;

    const vodButtonText = (vod && "Live Recording Embed") || "Live recording not found";
    const vodEmbed = vod && (
      <StyledLayout padding={{ top: 0.5 }} borderTop fullWidth data-test-selector={TestSelectors.VOD_EMBED}>
        <VodVideoEmbed vod={vod} />
      </StyledLayout>
    );

    const squadEmbed = squad && (
      <StyledLayout fullWidth borderTop textAlign={TextAlign.Center}>
        <CoreText bold>Squad Details</CoreText>
        <StyledLayout fullWidth display={Display.Flex} justifyContent={JustifyContent.Around}>
          <StyledLayout fullWidth borderRight>
            <FormGroup label="ID:"> {squad.id}</FormGroup>
          </StyledLayout>
          <StyledLayout fullWidth borderRight>
            <FormGroup label="Owner:"> {squad.owner && <User user={squad.owner} />}</FormGroup>
          </StyledLayout>
          <StyledLayout fullWidth borderRight>
            <FormGroup label="Members:">
              {squad.members &&
                squad.members.map(u => (
                  <StyledLayout key={u.id} display={Display.Flex} justifyContent={JustifyContent.Between}>
                    <User user={u} />
                    <VodPopoutButton
                      tooltip={"Vod of this user at the time of report"}
                      channelID={u.id}
                      timestamp={new Date(createdAt).getTime()}
                    />
                  </StyledLayout>
                ))}
            </FormGroup>
          </StyledLayout>
          <Layout fullWidth>
            <FormGroup label="Created At:">{squad.createdAt && new Date(squad.createdAt).toDisplayFormat()}</FormGroup>
          </Layout>
        </StyledLayout>
      </StyledLayout>
    );

    return (
      <StyledLayout fullWidth display={Display.Flex} flexDirection={FlexDirection.Column}>
        <Layout fullWidth display={Display.Flex} textAlign={TextAlign.Center}>
          <IndicatorButton
            data-track-click="report-user-embed-related-clips"
            type={ButtonType.Text}
            indicatorIcon={SVGAsset.Popout}
            fullWidth
            linkTo={similarClipsURL}
            size={ButtonSize.Small}
            targetBlank
          >
            Clips near report time
          </IndicatorButton>
          <StyledLayout borderRight />
          <StyledLayout fullWidth textAlign={TextAlign.Center} flexOrder={1} borderRight>
            <Button
              data-track-click="report-user-embed-show-vod"
              type={ButtonType.Text}
              fullWidth
              disabled={!vod}
              onClick={this.toggleShowVodEmbed}
              data-test-selector={TestSelectors.VOD_EMBED_TOGGLE_BUTTON}
            >
              {vodButtonText}
            </Button>
          </StyledLayout>
          <StyledLayout fullWidth textAlign={TextAlign.Center} flexOrder={1}>
            <IndicatorButton
              data-track-click="report-user-embed-live-chat"
              type={ButtonType.Text}
              indicatorIcon={SVGAsset.LogoGlitch}
              fullWidth
              disabled={true}
              linkTo={chatPopupURL}
              size={ButtonSize.Small}
              targetBlank
            >
              <CoreText color={Color.Alt2}>Live Chat (coming soon!)</CoreText>
            </IndicatorButton>
          </StyledLayout>
        </Layout>
        {!!this.showVodEmbed && vodEmbed}
        {squadEmbed}
      </StyledLayout>
    );
  }

  @action
  private toggleShowVodEmbed = () => {
    this.showVodEmbed = !this.showVodEmbed;
  };
}
