import * as React from "react";

import * as Enzyme from "enzyme";
import * as Adapter from "enzyme-adapter-react-16";
import { Router } from "react-router-dom";

import { DELETE_VOD_COMMENT, DeleteVodCommentButton } from "aegis/features/delete-vod-comment-button";
import { VodCommentReport } from "aegis/features/reports/models";
import { createHistory } from "aegis/functionality/utils/contrib/twilight/history";
import { mockSuccessResponse } from "aegis/tests/apollo";
import { wrapWithGQLProvider } from "aegis/tests/graphql";
import { waitSomeMs } from "aegis/tests/utils/wait-some-ms";
import { Button } from "twitch-core-ui";
import { NewVodCommentReport } from "./testutil";
import { TestSelectors, VodCommentEmbed } from "./vod-comment-embed";

import { ConfirmationModalConfirmSelector } from "aegis/functionality/components/confirmation-modal/component";
import "aegis/functionality/date-extensions";

Enzyme.configure({ adapter: new Adapter() });

const mockDeleteVodCommentSuccess = mockSuccessResponse(
  DELETE_VOD_COMMENT,
  {},
  {
    data: {
      deleteVodComment: {
        success: true,
        __typename: "DeleteVodCommentPayload"
      }
    }
  }
);

function setupMount(overrides?: Partial<VodCommentReport>) {
  const report = NewVodCommentReport(overrides ? overrides : {});
  if (!!report.vodComment) {
    mockDeleteVodCommentSuccess.request.variables = { id: report.vodComment.id };
  }
  return Enzyme.mount(
    <Router history={createHistory()}>
      {wrapWithGQLProvider(<VodCommentEmbed report={report} />, [mockDeleteVodCommentSuccess]).wrappedComponent}
    </Router>
  );
}

describe("VodCommentEmbed", () => {
  const modalRoot = document.createElement("div");
  modalRoot.setAttribute("id", "portal-modal-root");
  const body = document.querySelector("body");
  if (body) {
    body.appendChild(modalRoot);
  }

  const globalDiv = document.createElement("div");
  document.body.appendChild(globalDiv);

  it("renders ui parts", () => {
    const wrapper = setupMount();
    expect(wrapper.find({ "data-test-selector": TestSelectors.CommentDeletedIndicator }).length).toBe(0);
    expect(wrapper.find({ "data-test-selector": TestSelectors.VodURLButton })).toBeTruthy();
    expect(wrapper.find({ "data-test-selector": TestSelectors.DeleteVodCommentButton })).toBeTruthy();
    expect(wrapper.find({ "data-test-selector": TestSelectors.Messages })).toBeTruthy();
  });

  it("does not render if vod url is missing ", () => {
    const wrapper = setupMount({ vodURL: undefined });
    expect(wrapper.find({ "data-test-selector": TestSelectors.CommentDeletedIndicator }).length).toBe(0);
    expect(wrapper.find({ "data-test-selector": TestSelectors.VodURLButton }).length).toBe(0);
    expect(wrapper.find({ "data-test-selector": TestSelectors.DeleteVodCommentButton }).length).toBe(0);
    expect(wrapper.find({ "data-test-selector": TestSelectors.Messages }).length).toBe(0);
  });

  it("does not render if vod comment is missing ", () => {
    const wrapper = setupMount({ vodComment: undefined });
    expect(wrapper.find({ "data-test-selector": TestSelectors.CommentDeletedIndicator }).length).toBe(0);
    expect(wrapper.find({ "data-test-selector": TestSelectors.VodURLButton }).length).toBe(0);
    expect(wrapper.find({ "data-test-selector": TestSelectors.DeleteVodCommentButton }).length).toBe(0);
    expect(wrapper.find({ "data-test-selector": TestSelectors.Messages }).length).toBe(0);
  });

  it("disables delete vod comment button on successful deletion", async () => {
    const wrapper = setupMount();

    expect(wrapper.find(DeleteVodCommentButton).prop("disabled")).toBe(false);
    wrapper
      .find(DeleteVodCommentButton)
      .find(Button)
      .simulate("click");
    wrapper
      .find({ "data-test-selector": ConfirmationModalConfirmSelector })
      .find(Button)
      .simulate("click");

    await waitSomeMs();
    wrapper.update();

    expect(wrapper.find(DeleteVodCommentButton).prop("disabled")).toBe(true);
    expect(wrapper.find(DeleteVodCommentButton).text()).toContain("Vod Comment Deleted");
  });
});
