import * as React from "react";

import { action, observable } from "mobx";
import { observer } from "mobx-react";

import { DeleteVodCommentButton, DeleteVodCommentResponse } from "aegis/features/delete-vod-comment-button";
import { CollapsibleMessageLog, HighlightedRows, MessageColumn } from "aegis/features/messages";
import { HighlightType } from "aegis/features/messages/components/message-log-row";
import { VodCommentReport } from "aegis/features/reports";
import {
  AlignItems,
  Button,
  ButtonType,
  Color,
  CoreText,
  Display,
  Layout,
  SVG,
  SVGAsset,
  TextAlign,
  TextType
} from "twitch-core-ui";

export enum TestSelectors {
  CommentDeletedIndicator = "comment-deleted-indicator",
  VodURLButton = "vod-url-button",
  DeleteVodCommentButton = "delete-vod-comment-button",
  Messages = "messages"
}
export interface PublicProps {
  report: VodCommentReport;
}

type Props = PublicProps;

@observer
export class VodCommentEmbed extends React.Component<Props> {
  @observable private deleted: boolean = false;
  @observable private error: Error | null = null;

  public render() {
    const { vodURL, vodComment } = this.props.report;

    if (!vodURL) {
      return <CoreText color={Color.Error}>Malformed report data: vod missing</CoreText>;
    }

    if (!vodComment) {
      return <CoreText color={Color.Error}>Malformed report data: comment missing</CoreText>;
    }

    const commentDeletedIndicator = vodComment.isDeleted ? (
      <CoreText type={TextType.Span} data-test-selector={TestSelectors.CommentDeletedIndicator}>
        The comment is deleted
      </CoreText>
    ) : (
      ""
    );

    const highlights: HighlightedRows = {};
    highlights[vodComment.id] = HighlightType.Alt;

    return (
      <Layout>
        {commentDeletedIndicator}
        <CollapsibleMessageLog
          data-test-selector={TestSelectors.Messages}
          highlights={highlights}
          entries={[vodComment]}
          chatColumns={[MessageColumn.Time, MessageColumn.From, MessageColumn.Message]}
        />

        <Button
          data-track-click="report-vod-comment-embed-show-in-vod"
          type={ButtonType.Hollow}
          linkTo={vodURL}
          targetBlank={true}
          disabled={!vodURL}
          data-test-selector={TestSelectors.VodURLButton}
        >
          <Layout display={Display.Flex} alignItems={AlignItems.Center} textAlign={TextAlign.Center}>
            <CoreText>View in VOD &nbsp;</CoreText>
            <SVG asset={SVGAsset.Popout} />
          </Layout>
        </Button>
        <DeleteVodCommentButton
          id={vodComment.id}
          type={ButtonType.Alert}
          onCompleted={this.onCompleted}
          onError={this.onError}
          disabled={this.deleted}
          data-test-selector={TestSelectors.DeleteVodCommentButton}
        >
          <div className="button-text">{this.deleted ? "Vod Comment Deleted" : "Delete Comment"}</div>
        </DeleteVodCommentButton>
        {this.error ? <CoreText color={Color.Error}>{this.error.toString()}</CoreText> : null}
      </Layout>
    );
  }

  @action
  private onError = (error: Error) => {
    console.error("Failed to delete vod comment: ", error);
    this.error = error;
  };

  @action
  private onCompleted = (response: DeleteVodCommentResponse) => {
    this.deleted = response.deleteVodComment.success;
  };
}
