import * as React from "react";

import { Report } from "aegis/features/reports";
import { UserCardLink } from "aegis/features/user-card/user-link";
import { CopyLinkButton } from "aegis/functionality/components/copy-link-button";
import { copyTextToClipboard } from "aegis/functionality/utils/contrib/twilight/copy-text-to-clipboard";
import { getFullLanguageFromCode } from "aegis/functionality/utils/language-from-language-code";
import {
  AlignItems,
  ButtonIcon,
  ButtonIconSize,
  ButtonIconType,
  Color,
  CoreLink,
  CoreText,
  Display,
  FlexDirection,
  FlexWrap,
  FontSize,
  JustifyContent,
  Layout,
  StyledLayout,
  SVG,
  SVGAsset,
  SVGType,
  TextType,
  Tooltip,
  TooltipAlign,
  TooltipDirection
} from "twitch-core-ui";
import { ReportDescription } from "./report-description";

import { ReportAuditModalButton } from "../report-audits";
import "./styles.scss";

enum LanguageSetType {
  System = "system",
  Admin = "admin"
}

export enum TestSelectors {
  CopyIdButton = "copy-id-button",
  CopyLinkButton = "copy-link-button",
  ReportLanguage = "report-language",
  ReportTargetUser = "report-target-user"
}

export interface Props {
  report: Report;
  expanded?: boolean;
}

export class ReportHeader extends React.Component<Props> {
  public render() {
    const fullLanguage = getFullLanguageFromCode(this.props.report.language);
    const { targetUser } = this.props.report;
    const userName = !targetUser ? "Not Found" : targetUser.displayName;

    const copyReportIdToClipboard = (event: React.MouseEvent<HTMLElement>) => {
      event.stopPropagation();
      copyTextToClipboard(this.props.report.id);
    };

    const reportLanguage = `${this.props.report.language ? this.props.report.language.toUpperCase() : "??"}`;
    const createdAt = new Date(this.props.report.createdAt).toDisplayFormat();
    const reportUrl = `${location.protocol}//${location.host}/report/?id=${this.props.report.id}`;

    return (
      <StyledLayout
        padding={{ top: 1, bottom: 0.5 }}
        className={this.props.expanded ? "report-header__expanded" : "report-header"}
        borderTop
        borderLeft
        borderRight
      >
        <Layout fullWidth display={Display.Flex} flexWrap={FlexWrap.NoWrap} padding={{ x: 2 }}>
          <Layout flexGrow={0} display={Display.Flex} flexWrap={FlexWrap.NoWrap} className="report-header-items">
            <Layout
              className="report-header-item"
              display={Display.Flex}
              flexDirection={FlexDirection.Column}
              alignItems={AlignItems.Start}
            >
              <CoreText type={TextType.Span} color={Color.Alt2}>
                Report Type
              </CoreText>
              <Layout className="aegis-pill__content report-header-pill-size">
                {this.props.report.content.displayName}
              </Layout>
            </Layout>
            <Layout
              className="report-header-item"
              display={Display.Flex}
              flexDirection={FlexDirection.Column}
              alignItems={AlignItems.Start}
            >
              <CoreText type={TextType.Span} color={Color.Alt2}>
                Reason
              </CoreText>
              <Layout className="aegis-pill__reason report-header-pill-size">
                {this.props.report.reason.displayName}
              </Layout>
            </Layout>
            <Layout className="report-header-item" display={Display.Flex} flexDirection={FlexDirection.Column}>
              <CoreText type={TextType.Span} color={Color.Alt2}>
                Reported User
              </CoreText>
              <UserCardLink userID={targetUser ? targetUser.id : undefined}>
                <CoreLink>
                  <CoreText
                    fontSize={FontSize.Size4}
                    ellipsis
                    data-test-selector={TestSelectors.ReportTargetUser}
                    color={Color.Base}
                  >
                    {userName}
                  </CoreText>
                </CoreLink>
              </UserCardLink>
            </Layout>
          </Layout>
          <Layout
            className="header-timebox"
            flexGrow={1}
            display={Display.Flex}
            alignItems={AlignItems.Center}
            justifyContent={JustifyContent.End}
          >
            {this.getReportLanguageIcon()}

            <Tooltip label={fullLanguage ? fullLanguage : "Unknown"} direction={TooltipDirection.Top}>
              <CoreText
                type={TextType.Span}
                data-test-selector={TestSelectors.ReportLanguage}
                color={Color.Alt2}
                fontSize={FontSize.Size5}
              >
                {reportLanguage}
              </CoreText>
            </Tooltip>
            <Layout padding={{ x: 0.5 }}>
              <CoreText type={TextType.Span} color={Color.Alt2} fontSize={FontSize.Size5}>
                •
              </CoreText>
            </Layout>
            <CoreText type={TextType.Span} color={Color.Alt2} fontSize={FontSize.Size5} lines={1}>
              {createdAt}
            </CoreText>
            <Layout display={Display.Flex}>
              <div
                data-track-click="report-header-copy-report-id"
                className="report-header-id-box"
                onClick={copyReportIdToClipboard}
              >
                <div className="report-header-id-content">
                  <Tooltip align={TooltipAlign.Right} label="Copy report id">
                    <div className="id-icon">
                      <ButtonIcon
                        size={ButtonIconSize.Large}
                        ariaLabel="Copy report id"
                        icon={SVGAsset.NotificationInfo}
                        type={ButtonIconType.Secondary}
                        data-test-selector={TestSelectors.CopyIdButton}
                        tabIndex={-1}
                        overlay
                      />
                    </div>
                    <div className="id-background" />
                    <Layout
                      display={Display.Flex}
                      alignItems={AlignItems.Center}
                      justifyContent={JustifyContent.End}
                      padding={{ right: 0.5 }}
                    >
                      <CoreText fontSize={FontSize.Size5} ellipsis>
                        {this.props.report.id}
                      </CoreText>
                    </Layout>
                  </Tooltip>
                </div>
              </div>
            </Layout>
            <Layout display={Display.Flex}>
              <Tooltip align={TooltipAlign.Right} label="Copy link to Report">
                <CopyLinkButton link={reportUrl}>
                  <ButtonIcon
                    data-track-click="report-header-copy-report-link"
                    size={ButtonIconSize.Large}
                    ariaLabel="Copy link"
                    icon={SVGAsset.Link}
                    overlay
                    type={ButtonIconType.Secondary}
                    data-test-selector={TestSelectors.CopyLinkButton}
                    tabIndex={-1}
                  />
                </CopyLinkButton>
              </Tooltip>
            </Layout>
            <ReportAuditModalButton report={this.props.report} />
          </Layout>
        </Layout>
        <Layout margin={{ top: 1 }} padding={{ x: 2 }} display={Display.Flex} flexWrap={FlexWrap.NoWrap}>
          <ReportDescription report={this.props.report} expanded={this.props.expanded} />
          {!!this.props.children && <Layout flexGrow={1} padding={{ right: 0.5 }} />}
          <Layout padding={{ y: 0.5 }}>{this.props.children}</Layout>
        </Layout>
      </StyledLayout>
    );
  }

  private getReportLanguageIcon = () => {
    let languageType: LanguageSetType | undefined = undefined;
    const { report } = this.props;

    if (report.language) {
      languageType = this.props.report.audits.some(a => !!a.actionBy && a.action === "assign_language")
        ? LanguageSetType.Admin
        : LanguageSetType.System;
    }

    let tooltipLabel: string;
    switch (languageType) {
      case LanguageSetType.Admin:
        tooltipLabel = "Report language selected by Admin";
        break;
      case LanguageSetType.System:
        tooltipLabel = "Description language identified by System";
        break;
      default:
        tooltipLabel = "System could not detect language";
    }

    const SVGIcon = languageType === LanguageSetType.Admin ? SVGAsset.GlyphLive : SVGAsset.Automod;

    return (
      <StyledLayout display={Display.Flex} padding={{ right: 0.5 }} color={Color.Alt2}>
        <Tooltip label={tooltipLabel}>
          <SVG asset={SVGIcon} type={SVGType.Inherit} />
        </Tooltip>
      </StyledLayout>
    );
  };
}
