import * as React from "react";

import gql from "graphql-tag";

import { BanFormModal, Props as ModalProps } from "aegis/features/banform/banform-modal";
import { CurseWhisperReport, Report, WhisperReport } from "aegis/features/reports";
import { ReportFields, ReportSubtypeFields } from "aegis/fragments";
import { Query } from "aegis/functionality/apollo-wrapper";
import { errorNotification } from "aegis/functionality/error-notifications";
import { ReportContentCode, User } from "aegis/models";
import {
  SuspensionGuideContent,
  SuspensionGuideDetailedReason,
  SuspensionGuideReason
} from "aegis/models/suspension_guide";
import { LoadingSpinner } from "twitch-core-ui";

export interface Props {
  strikeUser: User;
  report: Report;
}

export type ReportSubtypeFieldsInput = {
  id: string;
};

export type ReportSubtypeFieldsResponse = {
  report: Report;
};

export const FETCH_REPORT_SUBTYPE_FIELDS = gql`
  query reportSubtypeFields($id: ID!) {
    report(id: $id) {
      ...ReportFields
      ...ReportSubtypeFields
    }
  }
  ${ReportFields}
  ${ReportSubtypeFields}
`;

// Thin wrapper on ban form modal that fetches report details before rendering the modal
export class ReportBanFormModalQuery extends Query<ReportSubtypeFieldsResponse, ReportSubtypeFieldsInput> {}

export const ReportBanFormModal: React.SFC<Props> = ({ strikeUser, report }) => {
  if (!report) {
    return null;
  }
  return (
    <ReportBanFormModalQuery
      displayName="ReportBanFormModal"
      onError={error =>
        errorNotification("Failed to fetch report details. Ignoring defaults ", {
          error: error
        })
      }
      query={FETCH_REPORT_SUBTYPE_FIELDS}
      fetchPolicy={"cache-first"}
      variables={{ id: report.id }}
      skip={!report}
    >
      {({ loading, data }) => {
        if (loading) {
          return <LoadingSpinner />;
        }

        return <BanFormModal {...defaultProps(strikeUser, { ...report, ...(data && data.report) })} />;
      }}
    </ReportBanFormModalQuery>
  );
};

const defaultProps = (strikeUser: User, report: Report) => {
  const banFormProps: ModalProps = {
    displayName: strikeUser.displayName,
    targetUserID: strikeUser.id,
    placeholders: {
      reportID: report.id
    }
  };

  switch (report.content.code) {
    case ReportContentCode.CurseWhisper:
      banFormProps.defaultContent = SuspensionGuideContent.Whisper;
      banFormProps.defaultReason = SuspensionGuideReason.Spam;
      banFormProps.defaultDetailedReason = SuspensionGuideDetailedReason.SpamBots;
      banFormProps.placeholders!.whisper = (report as CurseWhisperReport).rawText;
      break;
    case ReportContentCode.Whisper:
      banFormProps.defaultContent = SuspensionGuideContent.Whisper;
      banFormProps.defaultReason = SuspensionGuideReason.Spam;
      banFormProps.defaultDetailedReason = SuspensionGuideDetailedReason.SpamBots;
      banFormProps.placeholders!.whisper = (report as WhisperReport).rawText;
      break;
    case ReportContentCode.Vod:
      banFormProps.defaultContent = SuspensionGuideContent.Vod;
      break;
    default:
    // Do nothing
  }

  return banFormProps;
};
