import * as React from "react";

import { mount, shallow } from "enzyme";
import * as Enzyme from "enzyme";
import * as Adapter from "enzyme-adapter-react-16";
import { Router } from "react-router-dom";

import { FETCH_REPORT_CONTENTS } from "aegis/features/report-content-dropdown";
import { FETCH_REPORT_REASONS } from "aegis/features/report-reason-dropdown";
import { TestSelectors as HeaderSelectors } from "aegis/features/report-view/components/report-header";
import { ReportViewType } from "aegis/features/report-view/models";
import { mockReport } from "aegis/features/report-view/reportMocks";
import { createHistory } from "aegis/functionality/utils/contrib/twilight/history";
import { MockAdmin, MockUser } from "aegis/models/mocks";
import { MockReportContent, MockReportReason } from "aegis/models/mocks/report";
import { ReportStore } from "aegis/stores";
import { mockSuccessResponse } from "aegis/tests/apollo";
import { wrapWithGQLProvider } from "aegis/tests/graphql";
import { Props, ReportHotkeyLayerComponent, TestSelectors } from "./";

import { MockTrackingClient } from "aegis/features/tracking/client/mock";
import { FETCH_REPORT_ADMINS } from "aegis/functionality/components/admin-select";
import "aegis/functionality/date-extensions";

Enzyme.configure({ adapter: new Adapter() });

const setup = (overrides?: Partial<Props>) => {
  const reportStore = new ReportStore();
  reportStore.expand("fakeID"); // Sets a default "expanded" id so new report-views are collapsed
  const combinedProps: Props = {
    trackingClient: new MockTrackingClient(),
    report: mockReport({ fromUser: MockUser(), targetUser: MockUser() }),
    viewType: ReportViewType.Main,
    reportStore: reportStore,
    hideReport: jest.fn(),
    ...overrides
  };
  const wrapper = shallow(<ReportHotkeyLayerComponent.wrappedComponent {...combinedProps} />);
  return {
    props: combinedProps,
    wrapper,
    reportStore
  };
};

const mockReportAdmins = mockSuccessResponse(
  FETCH_REPORT_ADMINS,
  {},
  {
    data: {
      reportAdmins: [MockAdmin()]
    }
  }
);

const mockReportContentsSuccess = mockSuccessResponse(
  FETCH_REPORT_CONTENTS,
  {},
  {
    data: {
      reportContents: [MockReportContent()]
    }
  }
);

const mockReportReasonsSuccess = mockSuccessResponse(
  FETCH_REPORT_REASONS,
  {},
  {
    data: {
      reportReasons: [MockReportReason()]
    }
  }
);

const setupMount = (overrides?: Partial<Props>) => {
  const reportStore = new ReportStore();
  const combinedProps = {
    report: mockReport({ fromUser: MockUser(), targetUser: MockUser() }),
    viewType: ReportViewType.Main,
    reportStore: reportStore,
    hideReport: jest.fn(),
    ...overrides
  };
  const wrapper = mount(
    wrapWithGQLProvider(
      <Router history={createHistory()}>
        <ReportHotkeyLayerComponent {...combinedProps} />
      </Router>,
      [
        mockReportAdmins,
        mockReportAdmins,
        mockReportAdmins,
        mockReportContentsSuccess,
        mockReportContentsSuccess,
        mockReportContentsSuccess,
        mockReportReasonsSuccess,
        mockReportReasonsSuccess,
        mockReportReasonsSuccess
      ]
    ).wrappedComponent
  );
  return {
    props: combinedProps,
    wrapper,
    reportStore
  };
};

describe("Report View", () => {
  it("renders with a valid UserReport", () => {
    const { wrapper } = setup({});

    expect(wrapper).toMatchSnapshot();
  });

  it("renders when a report's targetUser is null", () => {
    const { wrapper } = setup({ report: mockReport() });

    expect(wrapper).toMatchSnapshot();
  });

  it("expands when clicked in body", () => {
    const { wrapper, reportStore } = setup();
    const { id } = mockReport();

    // Collapses the report (as the first report is now open by default)
    reportStore.collapse(id);
    wrapper.update();

    expect(wrapper.find({ "data-test-selector": TestSelectors.CollapsedReport })).toHaveLength(1);
    wrapper.find({ "data-test-selector": TestSelectors.CollapsedReport }).simulate("click");
    wrapper.update();

    expect(wrapper.find({ "data-test-selector": TestSelectors.CollapsedReport })).toHaveLength(0);
    expect(wrapper.find({ "data-test-selector": TestSelectors.ExpandedReport })).toHaveLength(1);
  });

  it("doesn't expand when a button is clicked", () => {
    const { wrapper, reportStore } = setupMount();
    const { id } = mockReport();

    // Collapses the report (as the first report is now open by default)
    reportStore.collapse(id);
    wrapper.update();

    const button = wrapper.find({ "data-test-selector": HeaderSelectors.CopyIdButton });
    button.at(1).simulate("click");
    wrapper.update();

    expect(wrapper.find({ "data-test-selector": TestSelectors.CollapsedReport })).toHaveLength(1);
    expect(wrapper.find({ "data-test-selector": TestSelectors.ExpandedReport })).toHaveLength(0);
  });
});
