import * as React from "react";

import { action, autorun, computed, observe } from "mobx";
import { inject, observer, Provider } from "mobx-react";
import { HotKeys } from "react-hotkeys";

import { ImageStore } from "aegis/features/images";
import { Report } from "aegis/features/reports";
import { EventName, HotkeyUsage } from "aegis/features/tracking";
import { TrackingClientProps } from "aegis/features/tracking/client";
import { PortalModal } from "aegis/functionality/components/portal-modal";
import { preventFireOnInput } from "aegis/functionality/utils/hotkey-handler-ignore-input";
import { User } from "aegis/models";
import { ReportStore } from "aegis/stores";
import { ReportActionContext, reportActionContextMapper, withReportActionContext } from "../../context";
import { ReportViewType } from "../../models";
import { ReportCard } from "../report-card";
import { ReportBanFormModal } from "./ban-form/container";

import "./styles.scss";

export interface PublicProps {
  report: Report;
  viewType: ReportViewType;
  reportStore: ReportStore;
  disableExpand?: boolean;
  strikeUser?: User;
  openStrikeForm?: (user: User) => void;
  closeStrikeForm?: () => void;
  prevReport?: () => void;
  nextReport?: () => void;
  hideReport?: () => void;
}

export type Props = PublicProps & TrackingClientProps;

export enum TestSelectors {
  CollapsedReport = "collapsed-report-view",
  ExpandedReport = "expanded-report-view"
}

/**
 * Parent "container" component for ReportCard, this component serves as
 * an interface between the ReportCard and the ReportStore, allowing the
 * card to focus on layout
 */
@inject("trackingClient")
@observer
export class ReportHotkeyLayerComponent extends React.Component<Props> {
  private screenshotStore = new ImageStore();
  private hotkeyAreaRef: HTMLElement | null;
  private hotkeyMap = {
    toggleCardExpansion: "o",
    closeReport: "esc",
    openStrikeForm: "s",
    nextReport: "j",
    prevReport: "k"
  };
  private hotkeyHandlers = {
    toggleCardExpansion: () => {
      this.onCardClick();
    },
    closeReport: () => {
      this.collapseCard();
    },
    openStrikeForm: () => {
      if (!this.props.report.targetUser || !this.props.openStrikeForm) {
        return;
      }
      this.props.openStrikeForm(this.props.report.targetUser!);
    },
    prevReport: () => {
      if (this.props.prevReport) {
        this.props.prevReport();
      }
    },
    nextReport: () => {
      if (this.props.nextReport) {
        this.props.nextReport();
      }
    }
  };

  // Declaring type for the wrapped component of the inject for tests to use
  static wrappedComponent: React.ComponentClass<Props>;

  @computed
  get expanded(): boolean {
    return this.props.reportStore.isExpanded(this.props.report.id);
  }

  get strikeForm(): JSX.Element | null {
    const { strikeUser, closeStrikeForm, report } = this.props;
    if (!strikeUser || !closeStrikeForm) {
      return null;
    }
    return (
      <PortalModal onClose={closeStrikeForm}>
        <ReportBanFormModal strikeUser={strikeUser} report={report} />
      </PortalModal>
    );
  }

  componentDidMount() {
    const { report, reportStore } = this.props;
    observe(reportStore, () => null);

    autorun(() => {
      if (reportStore.focusedID === report.id && this.hotkeyAreaRef) {
        this.hotkeyAreaRef.focus();
      }
    });
  }

  componentDidUpdate() {
    if (!this.props.strikeUser && this.props.reportStore.focusedID === this.props.report.id && this.hotkeyAreaRef) {
      this.hotkeyAreaRef.focus();
    }
  }

  componentWillUnmount() {
    this.props.reportStore.removeReport(this.props.report.id);
  }

  public render() {
    const { report, viewType, disableExpand } = this.props;
    return (
      <HotKeys keyMap={this.hotkeyMap} handlers={preventFireOnInput(this.withTrackingOnInput(this.hotkeyHandlers))}>
        <div
          tabIndex={0}
          ref={ref => (this.hotkeyAreaRef = ref)}
          onClick={this.expanded ? undefined : this.onCardClick}
          data-test-selector={this.expanded ? TestSelectors.ExpandedReport : TestSelectors.CollapsedReport}
          className="report-card-wrapper"
        >
          {this.strikeForm}
          <Provider screenshotStore={this.screenshotStore}>
            <ReportCard
              expanded={this.expanded}
              report={report}
              viewType={viewType}
              disableExpand={disableExpand}
              collapseCard={this.collapseCard}
              expandCard={this.expandCard}
              openStrikeForm={this.props.openStrikeForm!}
              onResolve={this.props.hideReport}
              hideReport={this.props.hideReport}
            />
          </Provider>
        </div>
      </HotKeys>
    );
  }

  @action
  private onCardClick = () => {
    const { report, reportStore } = this.props;
    reportStore.setFocusedID(report.id);
    reportStore.toggleExpanded(report.id);
  };

  @action
  private collapseCard = () => {
    const { report, reportStore, disableExpand } = this.props;
    if (!disableExpand) {
      reportStore.collapse(report.id);
    }
  };

  @action
  private expandCard = () => {
    const { report, reportStore } = this.props;
    reportStore.setFocusedID(report.id); // Auto focus the card we are expanding
    reportStore.expand(report.id);
  };

  private withTrackingOnInput(handler: { [key: string]: (keyEvent?: KeyboardEvent) => void }) {
    const newHandler: { [key: string]: (keyEvent?: KeyboardEvent) => void } = {};
    for (const key in handler) {
      if (handler.hasOwnProperty(key)) {
        newHandler[key] = this.withTracking(handler[key]);
      }
    }
    return newHandler;
  }

  private withTracking(run: (event: KeyboardEvent) => void) {
    const { report, trackingClient } = this.props;

    return (event: KeyboardEvent) => {
      if (!(event.target instanceof HTMLInputElement || event.target instanceof HTMLTextAreaElement)) {
        run(event);

        const e: HotkeyUsage = {
          report_id: report.id,
          hotkey: event.key
        };
        trackingClient!.Track(EventName.HotkeyUsage, e);
      }
    };
  }
}

export const ReportHotkeyLayer = withReportActionContext<ReportActionContext, Props>(reportActionContextMapper)(
  ReportHotkeyLayerComponent
);
