import * as React from "react";

import gql from "graphql-tag";

import { Helmet } from "react-helmet";

import { ReportQueueActionContextProvider } from "aegis/context/report";
import { LoadingFill } from "aegis/features/loading-fill";
import * as Fragments from "aegis/fragments";
import { Query } from "aegis/functionality/apollo-wrapper";
import { PollingInterval } from "aegis/models";
import { Color, CoreText } from "twitch-core-ui";
import { Report } from "../models";
import { AllHeldReportsComponent } from "./component";

export const FETCH_ALL_HELD_REPORTS_QUERY = gql`
  query allHeldReports($after: Int) {
    reportsOnHold(first: 10, after: $after) {
      edges {
        node {
          ...ReportFields
        }
      }
    }
  }
  ${Fragments.ReportFields}
`;

type Response = {
  reportsOnHold: {
    edges: ReportEdge[];
  };
};

type ReportEdge = {
  node: Report;
};

export class AllHeldReportsQuery extends Query<Response, {}> {}

export class AllHeldReports extends React.Component {
  public render() {
    return (
      <AllHeldReportsQuery displayName="AllHeldReports" query={FETCH_ALL_HELD_REPORTS_QUERY}>
        {({ client, loading, error, data, startPolling, stopPolling }) => {
          if (error) {
            console.error("Error fetching my reports", error);
            return <CoreText color={Color.Error}>An error occurred: {error.message}</CoreText>;
          }

          if (loading || !data) {
            return <LoadingFill />;
          }

          const { reportsOnHold } = data;
          const reports = reportsOnHold!.edges.map(edge => edge.node);

          if (reports.length === 0) {
            startPolling(PollingInterval.Normal);
          } else {
            stopPolling();
          }

          // Removes report from Apollo cache
          const hideReport = (report: Report) => {
            const cachedData = client.readQuery({ query: FETCH_ALL_HELD_REPORTS_QUERY }) as Response;
            if (!cachedData) {
              console.warn("No data for all holds. Skipping cache update");
              return;
            }
            cachedData.reportsOnHold.edges = cachedData.reportsOnHold.edges.filter(r => r.node.id !== report.id);
            client.writeQuery({ query: FETCH_ALL_HELD_REPORTS_QUERY, data: cachedData });
          };

          return (
            <ReportQueueActionContextProvider onCancelHold={hideReport}>
              <Helmet>
                <title>{`All Held Reports (${reports.length})`}</title>
              </Helmet>
              <AllHeldReportsComponent reportEntries={reports} hideReport={hideReport} />
            </ReportQueueActionContextProvider>
          );
        }}
      </AllHeldReportsQuery>
    );
  }
}
