import * as React from "react";

import gql from "graphql-tag";
import { graphql } from "react-apollo";
import { Helmet } from "react-helmet";

import { LoadingFill } from "aegis/features/loading-fill";
import { ReportViewType } from "aegis/features/report-view";
import * as Fragments from "aegis/fragments";
import { Color, CoreText } from "twitch-core-ui";
import { ReportsComponent } from "../component";
import { Report } from "../models";

export const AllReportsQuery = gql`
  query reports($first: Int, $after: Cursor) {
    reports(first: $first, after: $after) {
      edges {
        node {
          ...ReportFields
        }
        cursor
      }
    }
  }
  ${Fragments.ReportFields}
`;

type InputProps = {
  first?: number;
  after?: number;
};

type QueryVals = {
  first: number;
  after: number;
};

type Response = {
  reports: {
    edges: ReportEdge[];
  };
};

type ReportEdge = {
  node: Report;
};

const withAllReports = graphql<InputProps, Response, QueryVals>(AllReportsQuery, {
  options: ({ first, after }) => {
    return { variables: { first: first ? first : 10, after: after ? after : 0 } };
  }
});

export const AllReportsContainer = withAllReports(({ data }) => {
  const { loading, reports, error } = data!;
  if (loading) {
    return <LoadingFill />;
  }
  if (error) {
    console.error(error);
    return <CoreText color={Color.Error}>An error occurred: {error.message}</CoreText>;
  }
  return <ReportsComponent viewType={ReportViewType.Main} reportEntries={reports!.edges.map(edge => edge.node)} />;
});

export const AllReports: React.SFC = () => {
  return (
    <>
      <Helmet>
        <title>All Reports</title>
      </Helmet>
      <AllReportsContainer />
    </>
  );
};
