import * as React from "react";

import { autorun } from "mobx";
import { inject, observer } from "mobx-react";

import { ReportView, ReportViewType } from "aegis/features/report-view";
import { reportQueueId } from "aegis/pages/reportqueuepage";
import { ReportStoreProps } from "aegis/stores";
import { Report } from "./models";

import { Layout } from "twitch-core-ui";

import "./styles.scss";

export interface Props {
  viewType: ReportViewType;
  reportEntries: Report[];
  onRemove?: (report: Report) => void; // onRemove is invoked when a report is removed from th view
}

@inject("reportStore")
@observer
export class ReportsComponent extends React.Component<Props & ReportStoreProps> {
  private autoFocusDisposer = autorun(() => {
    const { reportEntries, reportStore } = this.props;
    const reportQueue = document.getElementById(reportQueueId);

    if (!reportStore) {
      return;
    }
    if (!reportStore.hasFocusedID && reportEntries.length > 0 && reportQueue) {
      reportQueue.scrollTo({ top: 0 });
      reportStore.expand(reportEntries[0].id);
      reportStore.setFocusedID(reportEntries[0].id);
    }
  });

  public componentWillMount() {
    this.props.reportStore!.clear();
  }

  public componentWillUnmount() {
    this.props.reportStore!.clear();
    this.autoFocusDisposer();
  }

  public render() {
    const { viewType, onRemove, reportEntries, reportStore } = this.props;
    const reportElements: JSX.Element[] = [];
    reportEntries.forEach(report => {
      const hideReport = () => {
        if (onRemove) {
          onRemove(report);
        }
      };
      reportElements.push(
        <Layout margin={{ y: 2 }} key={report.id}>
          <ReportView
            viewType={viewType}
            prevReport={this.prevReport}
            nextReport={this.nextReport}
            report={report}
            reportStore={reportStore!}
            key={report.id}
            hideReport={hideReport}
          />
        </Layout>
      );
    });

    return (
      <div className="reports">
        {reportElements}
        {this.props.children}
      </div>
    );
  }

  private prevReport = () => {
    const { reportEntries, reportStore } = this.props;

    const currentReportIndex = reportEntries.findIndex(report => report.id === reportStore!.focusedID);
    if (currentReportIndex > 0) {
      const nextReportID = reportEntries[currentReportIndex - 1].id;
      reportStore!.setFocusedID(nextReportID);
    }
  };

  private nextReport = () => {
    const { reportEntries, reportStore } = this.props;
    const currentReportIndex = reportEntries.findIndex(report => report.id === reportStore!.focusedID);
    if (currentReportIndex !== -1 && currentReportIndex + 1 < reportEntries.length) {
      const nextReportID = reportEntries[currentReportIndex + 1].id;
      reportStore!.setFocusedID(nextReportID);
    }
  };
}
