import * as React from "react";

import gql from "graphql-tag";
import { action, observable } from "mobx";
import { observer } from "mobx-react";
import * as QueryString from "query-string";

import { Helmet } from "react-helmet";

import { LoadingFill } from "aegis/features/loading-fill";
import { ReportViewType } from "aegis/features/report-view";
import * as Fragments from "aegis/fragments";
import { Query } from "aegis/functionality/apollo-wrapper";
import { AddPageToQueryString, PageFromQueryProps, PageQueryProps } from "aegis/functionality/utils/pagination";
import { PollingInterval } from "aegis/models";
import { OffsetPagedResponse } from "aegis/models/offset_pageinfo";
import { RouteComponentProps, withRouter } from "react-router";
import { Color, CoreText } from "twitch-core-ui";
import { Report, ReportFilter, ReportQueue } from "../models";
import { InvestigativeQueueComponent } from "./component";
import { InvestigativeReportsHeader } from "./header";

const DefaultPageSize = 10;

export const FETCH_INVESTIGATIVE_REPORTS = gql`
  query investigativeReport($after: Int, $filter: ReportFilter, $first: Int) {
    reports(first: $first, after: $after, filter: $filter) {
      edges {
        node {
          ...ReportFields
        }
      }
      pageInfo {
        ...OffsetPageInfoFields
      }
    }
  }

  ${Fragments.ReportFields}
  ${Fragments.OffsetPageInfoFields}
`;

export interface Props {
  page?: number;
}

export interface InvestigativeReportsInput {
  filter: ReportFilter;
  first?: number;
  after?: number;
}

export type InvestigativeReportsResponse = {
  reports: OffsetPagedResponse<Report>;
};

export class InvestigativeReportsQuery extends Query<InvestigativeReportsResponse, InvestigativeReportsInput> {}

@observer
class InvestigativeReportsContainer extends React.Component<
  InvestigativeReportsInput & Props & RouteComponentProps<{}>
> {
  @observable
  private filter: ReportFilter = {
    assignedToQueue: ReportQueue.Investigative,
    status: ["new", "open"]
  };

  get pageSize(): number {
    return this.props.first || DefaultPageSize;
  }

  public render() {
    const { page, history } = this.props;
    return (
      <InvestigativeReportsQuery
        displayName="InvestigativeReportsContainer"
        query={FETCH_INVESTIGATIVE_REPORTS}
        variables={{
          after: ((page || 1) - 1) * this.pageSize,
          filter: this.filter,
          first: this.pageSize
        }}
      >
        {({ loading, error, data, stopPolling, startPolling, refetch }) => {
          if (error) {
            console.error("An error occurred fetching investigative queue reports:", error);
            return <CoreText color={Color.Error}>An error occurred: {error.message}</CoreText>;
          }
          if (loading) {
            return <LoadingFill />;
          }
          if (!data || !data.reports) {
            console.error("Data/data.reports is missing. Data is ", data);
            return <CoreText color={Color.Error}>Data/reports is missing from data</CoreText>;
          }

          const reports: Report[] = data.reports.edges.map(edge => edge.node);
          const pageInfo = data.reports.pageInfo;

          if (reports.length === 0) {
            startPolling(PollingInterval.Slow);
          } else {
            stopPolling();
          }

          return (
            <>
              <Helmet>
                <title>{`Investigative Queue (${reports.length})`}</title>
              </Helmet>
              <InvestigativeReportsHeader
                reports={reports}
                updateFilter={this.updateFilter}
                currentFilter={this.filter}
              />
              <InvestigativeQueueComponent
                viewType={ReportViewType.Investigative}
                reportEntries={reports}
                fetchPage={(p: number) => AddPageToQueryString(history, p)}
                currentPage={1 + pageInfo.offset / this.pageSize} // pages start from 1, but offsets from 0
                totalPages={Math.ceil(pageInfo.total / this.pageSize)}
                onRemove={() => {
                  refetch();
                }}
              />
            </>
          );
        }}
      </InvestigativeReportsQuery>
    );
  }

  @action
  private updateFilter = (filter: Partial<ReportFilter>) => {
    this.filter = { ...this.filter, ...filter };
  };
}

export const InvestigativeReports = withRouter<InvestigativeReportsInput & RouteComponentProps<{}>>(props => {
  const pageProps: PageQueryProps = QueryString.parse(props.history.location.search);
  return <InvestigativeReportsContainer {...props} page={PageFromQueryProps(pageProps)} />;
});
