import * as React from "react";

import { action, observable } from "mobx";
import { observer } from "mobx-react";

import { LocalesPicker } from "aegis/features/language-lookup-input";
import { QueueHeader } from "../header";
import { Report, ReportFilter } from "../models";

import {
  AlignItems,
  Background,
  Balloon,
  BalloonDirection,
  Button,
  ButtonSize,
  ButtonType,
  CoreText,
  Display,
  FlexWrap,
  Layout,
  Position,
  StyledLayout,
  TextTransform,
  TextType
} from "twitch-core-ui";

export interface PublicProps {
  reports: Report[];
  updateFilter: (filter: Partial<ReportFilter>) => void;
  currentFilter: ReportFilter;
}

@observer
export class InvestigativeReportsHeader extends React.Component<PublicProps> {
  @observable showLanguagePicker: boolean = false;

  public render() {
    const { reports, currentFilter } = this.props;
    const { language } = currentFilter;
    const displayLanguage = language ? <CoreText transform={TextTransform.Uppercase}>{language}</CoreText> : "None";

    const clearLanguage = () => this.onLanguageSelect(null);
    return (
      <QueueHeader>
        <CoreText type={TextType.H3}>Showing {reports.length} Report(s)</CoreText>

        <StyledLayout background={Background.Base} padding={{ y: 1 }}>
          <Layout display={Display.Flex} alignItems={AlignItems.Center} position={Position.Relative}>
            <CoreText>Language: </CoreText>
            <Layout padding={{ left: 1 }} />
            <Layout display={Display.Flex} flexWrap={FlexWrap.NoWrap}>
              <Button
                data-track-click="investigative-queue-show-language-dropdown"
                dropdown
                size={ButtonSize.Small}
                type={ButtonType.Hollow}
                onClick={this.toggleDropdown}
              >
                {displayLanguage}
              </Button>
              <Layout padding={{ left: 1 }} />
              {language && (
                <Button
                  data-track-click="investigative-queue-clear-language"
                  onClick={clearLanguage}
                  size={ButtonSize.Small}
                  type={ButtonType.Hollow}
                >
                  Clear
                </Button>
              )}
            </Layout>

            <Balloon show={this.showLanguagePicker} noTail direction={BalloonDirection.Bottom}>
              <Layout padding={1}>
                <LocalesPicker onSelect={this.onLanguageSelect} />
              </Layout>
            </Balloon>
          </Layout>
        </StyledLayout>
      </QueueHeader>
    );
  }

  @action
  private toggleDropdown = () => {
    this.showLanguagePicker = !this.showLanguagePicker;
  };

  @action
  private hideDropdown = () => {
    this.showLanguagePicker = false;
  };

  private onLanguageSelect = (code: string | null) => {
    this.props.updateFilter({ language: code ? code : undefined });
    this.hideDropdown();
  };
}
