import * as React from "react";

import gql from "graphql-tag";
import { observable } from "mobx";
import { inject, observer } from "mobx-react";
import * as QueryString from "query-string";

import { Helmet } from "react-helmet";

import { LoadingFill } from "aegis/features/loading-fill";
import { ReportViewType } from "aegis/features/report-view";
import * as Fragments from "aegis/fragments";
import { Query } from "aegis/functionality/apollo-wrapper";
import { AddPageToQueryString, PageFromQueryProps, PageQueryProps } from "aegis/functionality/utils/pagination";
import { PollingInterval } from "aegis/models";
import { OffsetPagedResponse } from "aegis/models/offset_pageinfo";
import { AuthStoreProps } from "aegis/stores";
import { RouteComponentProps, withRouter } from "react-router";
import { Color, CoreText } from "twitch-core-ui";
import { Report, ReportFilter } from "../models";
import { MyAssignedReportsComponent } from "./component";

const DefaultPageSize = 10;

export const FETCH_MY_ASSIGNED_REPORTS = gql`
  query myAssignedReport($first: Int, $after: Int, $filter: ReportFilter) {
    reports(first: $first, after: $after, filter: $filter) {
      edges {
        node {
          ...ReportFields
        }
      }
      pageInfo {
        ...OffsetPageInfoFields
      }
    }
  }

  ${Fragments.ReportFields}
  ${Fragments.OffsetPageInfoFields}
`;

export interface MyAssignedReportsInput {
  page?: number;
  filter: ReportFilter;
  first?: number;
}

export interface MyAssignedQueryVals {
  after?: number;
  filter: ReportFilter;
  first?: number;
}

export type MyAssignedReportsResponse = {
  reports: OffsetPagedResponse<Report>;
};

export class MyAssignedReportsQuery extends Query<MyAssignedReportsResponse, MyAssignedQueryVals> {}

type Props = MyAssignedReportsInput & AuthStoreProps & RouteComponentProps<{}>;

@inject("authStore")
@observer
export class MyAssignedReportsContainer extends React.Component<Props> {
  componentDidMount() {
    observable(this.props.authStore!);
  }

  get pageSize(): number {
    return this.props.first || DefaultPageSize;
  }

  public render() {
    const { authStore, page, history } = this.props;

    if (!authStore!.hasLoggedInUser) {
      return <CoreText color={Color.Error}>You must be logged in to see this page</CoreText>;
    }

    const filter: ReportFilter = { status: ["open"], assignedTo: authStore!.loggedInUser!.ldap };

    const queryVariable: MyAssignedQueryVals = {
      after: ((page || 1) - 1) * this.pageSize,
      filter: filter,
      first: this.pageSize
    };

    return (
      <>
        <Helmet>
          <title>My Assigned Reports</title>
        </Helmet>
        <MyAssignedReportsQuery
          displayName="MyAssignedReportsContainer"
          query={FETCH_MY_ASSIGNED_REPORTS}
          variables={queryVariable}
        >
          {({ client, loading, error, data, stopPolling, startPolling }) => {
            if (error) {
              console.error("An error occurred fetching my assigned reports:", error);
              return <CoreText color={Color.Error}>An error occurred: {error.message}</CoreText>;
            }
            if (loading) {
              return <LoadingFill />;
            }
            if (!data || !data.reports) {
              console.error("Data or data.reports is missing. Data is ", data);
              return <CoreText color={Color.Error}>Data or data.reports is missing from data</CoreText>;
            }

            const reports: Report[] = data.reports.edges.map(edge => edge.node);
            const pageInfo = data.reports.pageInfo;

            const move = (p: number) => {
              const after = (p - 1) * this.pageSize; // pages start from 1, but offsets from 0
              queryVariable.after = after;
              AddPageToQueryString(history, p);
            };

            const hideReport = (report: Report) => {
              const d = client.readQuery({
                query: FETCH_MY_ASSIGNED_REPORTS,
                variables: queryVariable
              }) as MyAssignedReportsResponse;
              if (!d) {
                console.warn(
                  "Data for FETCH_MY_REPORTS_QUERY query not found. Skipping cache update on removing report ",
                  report
                );
                return;
              }
              d.reports.edges = d.reports.edges.filter(r => report.id !== r.node.id);

              client.writeQuery({ query: FETCH_MY_ASSIGNED_REPORTS, variables: queryVariable, data: d });
            };

            if (reports.length === 0) {
              startPolling(PollingInterval.Normal);
            } else {
              stopPolling();
            }

            return (
              <MyAssignedReportsComponent
                viewType={ReportViewType.Main}
                reportEntries={reports}
                fetchPage={move}
                currentPage={1 + pageInfo.offset / this.pageSize} // pages start from 1, but offsets from 0
                totalPages={Math.ceil(pageInfo.total / this.pageSize)}
                onRemove={hideReport}
              />
            );
          }}
        </MyAssignedReportsQuery>
      </>
    );
  }
}

export const MyAssignedReports = withRouter<MyAssignedReportsInput & RouteComponentProps<{}>>(props => {
  const pageProps: PageQueryProps = QueryString.parse(props.history.location.search);
  return <MyAssignedReportsContainer {...props} page={PageFromQueryProps(pageProps)} />;
});
