import * as React from "react";

import gql from "graphql-tag";

import { Helmet } from "react-helmet";

import { ReportQueueActionContextProvider } from "aegis/context/report";
import { LoadingFill } from "aegis/features/loading-fill";
import * as Fragments from "aegis/fragments";
import { Query } from "aegis/functionality/apollo-wrapper";
import { Color, CoreText } from "twitch-core-ui";
import { Report } from "../models";
import { MyHeldReportsComponent } from "./component";

export const FETCH_MY_HELD_REPORTS = gql`
  query myHeldReports($after: Int) {
    me {
      id
      admin {
        id
        reportsOnHold(first: 10, after: $after) {
          edges {
            node {
              ...ReportFields
            }
          }
        }
      }
    }
  }
  ${Fragments.ReportFields}
`;

type Response = {
  me: {
    admin: {
      reportsOnHold: {
        edges: ReportEdge[];
      };
    };
  };
};

type ReportEdge = {
  node: Report;
};

export class MyHeldReportsQuery extends Query<Response, {}> {}

export class MyHeldReports extends React.Component {
  public render() {
    return (
      <MyHeldReportsQuery displayName="MyHeldReports" query={FETCH_MY_HELD_REPORTS}>
        {({ client, loading, error, data, refetch }) => {
          if (error) {
            console.error("Error fetching my reports", error);
            return <CoreText color={Color.Error}>An error occurred: {error.message}</CoreText>;
          }

          if (loading || !data) {
            return <LoadingFill />;
          }

          const { me } = data;

          const myReports = me!.admin.reportsOnHold.edges.map(edge => edge.node);

          // Removes report from Apollo cache
          const hideReport = (report: Report) => {
            const cachedData = client.readQuery({ query: FETCH_MY_HELD_REPORTS }) as Response;
            if (!cachedData) {
              console.warn("No data for my holds");
              return;
            }
            cachedData.me.admin.reportsOnHold.edges = cachedData.me.admin.reportsOnHold.edges.filter(
              r => r.node.id !== report.id
            );
            client.writeQuery({ query: FETCH_MY_HELD_REPORTS, data: cachedData });
          };

          // Clears all held reports from cache
          const refetchHolds = () => {
            const cachedData = client.readQuery({ query: FETCH_MY_HELD_REPORTS }) as Response;
            if (!cachedData) {
              console.warn("Data for FETCH_MY_HELD_REPORTS query not found. Skipping cache update for reload");
              return;
            }
            cachedData.me.admin.reportsOnHold.edges = [];

            client.writeQuery({ query: FETCH_MY_HELD_REPORTS, data: cachedData });
            refetch();
          };

          return (
            <ReportQueueActionContextProvider onCancelHold={hideReport}>
              <Helmet>
                <title>{`My Held Reports (${myReports.length})`}</title>
              </Helmet>
              <MyHeldReportsComponent reportEntries={myReports} hideReport={hideReport} reload={refetchHolds} />
            </ReportQueueActionContextProvider>
          );
        }}
      </MyHeldReportsQuery>
    );
  }
}
