import * as React from "react";

import { autorun } from "mobx";
import { observer } from "mobx-react";

import { ReportViewType } from "aegis/features/report-view";
import { ReportsComponent } from "../component";
import { Report } from "../models";
import { Header } from "./header";

import {
  AlignItems,
  Button,
  Color,
  CoreText,
  Display,
  FlexDirection,
  JustifyContent,
  Layout,
  LoadingSpinner,
  SpinnerSize,
  TextType
} from "twitch-core-ui";

export const DISPLAY_LIMIT = 20; // This should be removed when we resolve perf issues

export interface Props {
  reports: Report[];
  onEmpty: () => void;
  reload: () => void;
  hideReport: (r: Report) => void;
}

export enum TestSelectors {
  RefetchButton = "my-report--refetch-button",
  Reports = "my-report--reports"
}

@observer
export class MyReportsComponent extends React.Component<Props> {
  private onEmptyDisposer = autorun(() => {
    const { onEmpty, reports } = this.props;
    if (reports.length === 0) {
      onEmpty();
    }
  });

  public componentWillUnmount() {
    this.onEmptyDisposer();
  }

  public render() {
    const { reports, reload, hideReport } = this.props;

    const refetchButton = (
      <Layout padding={{ bottom: 1 }} fullWidth display={Display.Flex} justifyContent={JustifyContent.Center}>
        <Button data-track-click="my-reports-refetch" data-test-selector={TestSelectors.RefetchButton} onClick={reload}>
          Reload Queue
        </Button>
      </Layout>
    );

    const reportComponent =
      reports.length === 0 ? (
        <Layout
          fullWidth
          padding={{ y: 1 }}
          display={Display.Flex}
          alignItems={AlignItems.Center}
          justifyContent={JustifyContent.Center}
          flexDirection={FlexDirection.Column}
        >
          <Layout display={Display.Inline} padding={1}>
            <CoreText type={TextType.H4} color={Color.Alt2}>
              No reports in queue, fetching more now...
            </CoreText>
          </Layout>
          <Layout padding={{ top: 1, bottom: 2 }}>
            <LoadingSpinner size={SpinnerSize.Large} />
          </Layout>
          {refetchButton}
        </Layout>
      ) : (
        <ReportsComponent
          data-test-selector={TestSelectors.Reports}
          viewType={ReportViewType.Main}
          reportEntries={reports.slice(0, DISPLAY_LIMIT)}
          onRemove={hideReport}
        >
          {refetchButton}
        </ReportsComponent>
      );

    return (
      <>
        <Header reports={reports} />
        {reportComponent}
      </>
    );
  }
}
