import * as React from "react";

import gql from "graphql-tag";

import { Helmet } from "react-helmet";

import { ReportQueueActionContextProvider } from "aegis/context/report";
import * as Fragments from "aegis/fragments";
import { PollingInterval } from "aegis/models";
import { Report } from "../models";
import { MyReportsComponent } from "./component";

import { Query } from "aegis/functionality/apollo-wrapper";
import { getMemoryUsedPercentage, memoryUsageWarning } from "aegis/functionality/utils/memory-usage";
import { Color, CoreText } from "twitch-core-ui";

export const FETCH_MY_REPORTS_QUERY = gql`
  query myReports {
    me {
      id
      admin {
        id
        reports(filters: [Assigned, Routed]) {
          ...ReportFields
        }
      }
    }
  }
  ${Fragments.ReportFields}
`;

export type MyReportResponse = {
  me: {
    admin: {
      reports: Report[];
    };
  };
};

export class MyReportsQuery extends Query<MyReportResponse, {}> {}

export class MyReports extends React.Component {
  public render() {
    return (
      <MyReportsQuery displayName="MyReports" query={FETCH_MY_REPORTS_QUERY} fetchPolicy="network-only">
        {({ client, loading, error, data, refetch, startPolling, stopPolling }) => {
          if (error) {
            console.error("Error fetching my reports", error);
            return <CoreText color={Color.Error}>An error occurred: {error.message}</CoreText>;
          }

          let reports: Report[] = [];
          // Optimistic loading. Show view with 0 reports instead of a loading fill
          if (!loading && data && data.me.admin) {
            reports = data.me.admin.reports;

            if (reports.length === 0) {
              startPolling(PollingInterval.Normal);
            } else {
              stopPolling();
            }
          }

          const hideReport = (report: Report) => {
            // Temp fix for memory overuse in Report Queue, recommends a refresh when memory use goes over 70%
            if (getMemoryUsedPercentage() > 0.7) {
              memoryUsageWarning();
            }
            const d = client.readQuery({ query: FETCH_MY_REPORTS_QUERY }) as MyReportResponse;
            if (!d) {
              console.warn(
                "Data for FETCH_MY_REPORTS_QUERY query not found. Skipping cache update on removing report ",
                report
              );
              return;
            }
            d.me.admin.reports = d.me.admin.reports.filter(r => report.id !== r.id);

            client.writeQuery({ query: FETCH_MY_REPORTS_QUERY, data: d });
          };

          const refetchReports = () => {
            const d = client.readQuery({ query: FETCH_MY_REPORTS_QUERY }) as MyReportResponse;
            if (!d) {
              console.warn("Data for FETCH_MY_REPORTS_QUERY query not found. Skipping cache update for reload");
              return;
            }
            d.me.admin.reports = [];

            client.writeQuery({ query: FETCH_MY_REPORTS_QUERY, data: d });
            refetch();
          };

          return (
            <>
              <Helmet>
                <title>{`My Queue (${reports.length})`}</title>
              </Helmet>
              <ReportQueueActionContextProvider onAutoResolve={refetchReports}>
                <MyReportsComponent
                  reports={reports}
                  onEmpty={refetch}
                  reload={refetchReports}
                  hideReport={hideReport}
                />
              </ReportQueueActionContextProvider>
            </>
          );
        }}
      </MyReportsQuery>
    );
  }
}
