import * as React from "react";

import gql from "graphql-tag";

import { RouteComponentProps, withRouter } from "react-router";

import { LoadingFill } from "aegis/features/loading-fill";
import { ReportViewType } from "aegis/features/report-view";
import * as Fragments from "aegis/fragments";
import { Query } from "aegis/functionality/apollo-wrapper";
import { AddPageToQueryString } from "aegis/functionality/utils/pagination";
import { OffsetPagedResponse } from "aegis/models/offset_pageinfo";
import { Color, CoreText } from "twitch-core-ui";
import { Report, ReportFilter, ReportQueue } from "../models";
import { PagedReportsComponent } from "../paged-reports";
import { ReportSearchInput } from "./report-search-input/component";

const DefaultPageSize = 10;

export const FETCH_REPORT_SEARCH = gql`
  query reports($first: Int, $after: Int, $filter: ReportFilter) {
    reports(first: $first, after: $after, filter: $filter, sortAscending: false) {
      edges {
        node {
          ...ReportFields
        }
        offset
      }
      pageInfo {
        ...OffsetPageInfoFields
      }
    }
  }
  ${Fragments.ReportFields}
  ${Fragments.OffsetPageInfoFields}
`;

export interface ReportSearchQueryProps {
  targetUserID?: string;
  targetUserLogin?: string;
  targetUserRole?: string;
  fromUserID?: string;
  fromUserLogin?: string;
  fromUserRole?: string;
  content?: string;
  reason?: string;
  description?: string;
  status?: string;
  timeStart?: string;
  timeEnd?: string;
  assignedTo?: string;
  language?: string;
  assignedToQueue?: string;
  page?: string;
}

type QueryVals = {
  first: number;
  after: number;
  filter: ReportFilter;
};

type Response = {
  reports: OffsetPagedResponse<Report>;
};

export class ReportSearchQuery extends Query<Response, QueryVals> {}

type Props = RouteComponentProps<Response> & { queryProps: ReportSearchQueryProps; first?: number };

class ReportSearchContainer extends React.Component<Props> {
  public render() {
    const { history, queryProps, first } = this.props;
    const page = parseInt(queryProps.page || "1", 10);
    const filter = createFilterFromProps(queryProps);
    const pageSize = first || DefaultPageSize;
    return (
      <ReportSearchQuery
        displayName="ReportSearchContainer"
        query={FETCH_REPORT_SEARCH}
        variables={{ first: pageSize, after: ((page || 1) - 1) * pageSize, filter: filter }}
      >
        {({ loading, error, data, refetch }) => {
          let pageContents: React.ReactNode = null;
          if (loading) {
            pageContents = <LoadingFill />;
          } else if (error) {
            console.error(error);
            pageContents = <CoreText color={Color.Error}>An error occurred: {error.message}</CoreText>;
          } else if (data) {
            const { reports } = data;
            const pageInfo = reports.pageInfo;
            pageContents = (
              <PagedReportsComponent
                viewType={ReportViewType.Search}
                reportEntries={reports.edges.map(edge => edge.node)}
                currentPage={1 + pageInfo.offset / pageSize} // pages start from 1, but offsets from 0
                totalPages={Math.ceil(pageInfo.total / pageSize)}
                fetchPage={(p: number) => AddPageToQueryString(history, p)}
              />
            );
          }

          return (
            <>
              <ReportSearchInput queryProps={queryProps} refetch={this.onRefetch(refetch)} />
              {pageContents}
            </>
          );
        }}
      </ReportSearchQuery>
    );
  }

  private onRefetch = (refetch: (variables?: QueryVals | undefined) => {}) => {
    return async () => {
      const { queryProps, first } = this.props;
      const page = parseInt(queryProps.page || "1", 10);
      const filter = createFilterFromProps(queryProps);
      const pageSize = first || DefaultPageSize;
      await refetch({ first: pageSize, after: ((page || 1) - 1) * pageSize, filter: filter });
      this.setState(this.state); // trigger a re-render; refetch doesn't do it
    };
  };
}

export const ReportSearch = withRouter((props: Props) => {
  return <ReportSearchContainer {...props} />;
});

function createFilterFromProps(props: ReportSearchQueryProps): ReportFilter {
  return {
    status: props.status ? props.status.split(",") : [],
    assignedTo: props.assignedTo,
    assignedToQueue:
      props.assignedToQueue && props.assignedToQueue in ReportQueue
        ? (props.assignedToQueue as ReportQueue)
        : undefined,
    fromUserID: props.fromUserID,
    fromUserRole: props.fromUserRole,
    targetUserID: props.targetUserID,
    targetUserRole: props.targetUserRole,
    content: props.content,
    reason: props.reason,
    language: props.language,
    description: props.description,
    startTime: props.timeStart ? Math.round(parseInt(props.timeStart, 10) / 1000) : undefined,
    endTime: props.timeEnd ? Math.round(parseInt(props.timeEnd, 10) / 1000) : undefined
  };
}
