import * as React from "react";

import * as classnames from "classnames";
import * as QueryString from "query-string";
import { RouteComponentProps, withRouter } from "react-router";

import { AdminSelect } from "aegis/functionality/components/admin-select";
import { UserInput } from "aegis/functionality/components/user-input";
import { removeTZOffset, toDateTimeLocal } from "aegis/functionality/utils/timestamp-conversions";
import { Admin, User, UserRoleTagMapping } from "aegis/models";
import { Button, Input, InputType, Layout, Select } from "twitch-core-ui";
import { ReportSearchQueryProps } from "../";

import { ReportContentSelector } from "./components/content-select";
import { ReportReasonSelector } from "./components/reason-select";
import { ReportStatusSelector } from "./components/status-select";
import "./styles.scss";

export interface PublicProps {
  queryProps: ReportSearchQueryProps;
  refetch: () => void;
}

type Props = RouteComponentProps<{}> & PublicProps;

export class ReportSearchInputComponent extends React.Component<Props, ReportSearchQueryProps> {
  public constructor(props: Props) {
    super(props);

    this.state = props.queryProps;
  }

  public render() {
    // These classes are pulled from CoreUI to style the dateTime inputs
    // to match the CoreUI <Input> component
    const dtClasses = classnames(
      "tw-block",
      "tw-border-radius-medium",
      "tw-font-size-6",
      "tw-full-width",
      "tw-pd-l-1",
      "tw-pd-r-1",
      "tw-pd-y-05",
      "tw-input",
      {
        "tw-input--error": false
      }
    );
    const userRoleOptions: JSX.Element[] = [<option key="none" />];
    UserRoleTagMapping.forEach(({ text }, key) => {
      userRoleOptions.push(
        <option value={key} key={key}>
          {text}
        </option>
      );
    });
    const { queryProps } = this.props;
    const dtNow = Date.now();

    return (
      <Layout>
        <Layout className="report-search-input-grid" fullWidth>
          <Layout>
            Target User:{" "}
            <UserInput
              name="target-user"
              onChange={this.updateTargetUser}
              defaultID={queryProps.targetUserID}
              defaultLogin={!queryProps.targetUserID ? queryProps.targetUserLogin : undefined}
            />
          </Layout>
          <Layout>
            Target User Role:{" "}
            <Select onChange={this.updateTargetUserRole} value={this.state.targetUserRole}>
              {userRoleOptions}
            </Select>
          </Layout>
          <Layout>
            From User:{" "}
            <UserInput
              name="from-user"
              onChange={this.updateFromUser}
              defaultID={queryProps.fromUserID}
              defaultLogin={!queryProps.fromUserID ? queryProps.fromUserLogin : undefined}
            />
          </Layout>
          <Layout>
            From User Role:{" "}
            <Select onChange={this.updateFromUserRole} value={this.state.fromUserRole}>
              {userRoleOptions}
            </Select>
          </Layout>
          <Layout padding={{ bottom: 2 }}>
            Content: <ReportContentSelector onSelect={this.updateContent} defaultValue={queryProps.content} />
          </Layout>
          <Layout padding={{ bottom: 2 }}>
            Reason: <ReportReasonSelector onSelect={this.updateReason} defaultValue={queryProps.reason} />
          </Layout>
          <Layout padding={{ bottom: 2 }}>
            Description{" "}
            <Input type={InputType.Text} onChange={this.updateDescription} defaultValue={queryProps.description} />
          </Layout>
          <Layout padding={{ bottom: 2 }}>
            Status: <ReportStatusSelector onSelect={this.updateStatus} defaultValue={queryProps.status} />
          </Layout>
          <Layout padding={{ bottom: 2 }}>
            Start Time:
            <input
              type="datetime-local"
              className={dtClasses}
              onChange={(e: React.FormEvent<HTMLInputElement>): void => {
                const newVal = removeTZOffset(Date.parse(e.currentTarget.value)).toString();
                this.setState({ timeStart: newVal });
              }}
              defaultValue={toDateTimeLocal(queryProps.timeStart || "0", 0)}
            />
          </Layout>
          <Layout padding={{ bottom: 2 }}>
            End Time:
            <input
              type="datetime-local"
              className={dtClasses}
              onChange={(e: React.FormEvent<HTMLInputElement>): void => {
                const newVal = removeTZOffset(Date.parse(e.currentTarget.value)).toString();
                this.setState({ timeEnd: newVal });
              }}
              defaultValue={toDateTimeLocal(queryProps.timeEnd || dtNow.toString(), dtNow)}
            />
          </Layout>
          <Layout padding={{ bottom: 2 }}>
            Assigned To:{" "}
            <AdminSelect onSelect={this.updateAdmin} defaultValue={queryProps.assignedTo} floatSuggestions />
          </Layout>
        </Layout>
        <Button onClick={this.search}>Search</Button>
      </Layout>
    );
  }

  private search = () => {
    const currentParams = location.search;
    const queryParams = QueryString.parse(currentParams) as ReportSearchQueryProps;
    let updatedParams = QueryString.stringify({ ...queryParams, ...this.state });

    // currentParams contains ?-prefix
    const paramsChanged = currentParams.substring(1) !== updatedParams;

    if (paramsChanged) {
      updatedParams = QueryString.stringify({ ...queryParams, ...this.state, page: undefined });
    }

    this.props.history.push({ pathname: "/reportsearch", search: updatedParams });

    if (!paramsChanged) {
      this.props.refetch();
    }
  };

  private updateTargetUser = (user?: User) => {
    this.setState({ targetUserID: user && user.id, targetUserLogin: undefined }, () => {
      this.search();
    });
  };

  private updateTargetUserRole = (e: React.FormEvent<HTMLSelectElement>) => {
    const selected = e.currentTarget.value;
    if (selected && selected !== "") {
      this.setState({ targetUserRole: selected });
    } else {
      this.setState({ targetUserRole: undefined });
    }
  };

  private updateFromUser = (user?: User) => {
    this.setState({ fromUserID: user && user.id, fromUserLogin: undefined }, () => {
      this.search();
    });
  };

  private updateFromUserRole = (e: React.FormEvent<HTMLSelectElement>) => {
    const selected = e.currentTarget.value;
    if (selected && selected !== "") {
      this.setState({ fromUserRole: selected });
    } else {
      this.setState({ fromUserRole: undefined });
    }
  };

  private updateContent = (content: string) => {
    if (content === "") {
      this.setState({ content: undefined });
    } else {
      this.setState({ content: content });
    }
  };

  private updateReason = (reason: string) => {
    if (reason === "") {
      this.setState({ reason: undefined });
    } else {
      this.setState({ reason: reason });
    }
  };

  private updateStatus = (status: string) => {
    if (status === "") {
      this.setState({ status: undefined });
    } else {
      this.setState({ status: status });
    }
  };

  private updateDescription = (e: React.FormEvent<HTMLInputElement>) => {
    this.setState({ description: e.currentTarget.value });
  };

  private updateAdmin = (admin?: Admin) => {
    this.setState({ assignedTo: admin && admin.ldap }, () => {
      this.search();
    });
  };
}

export const ReportSearchInput: React.ComponentClass<PublicProps> = withRouter<Props>(ReportSearchInputComponent);
