import * as React from "react";

import * as history from "history";
import { observer } from "mobx-react";
import { withRouter } from "react-router";

import { Shift, ShiftType } from "aegis/models";
import { Color, CoreText, FontSize, Layout, Tab, TabWrapper } from "twitch-core-ui";

interface TabProps {
  active?: boolean;
  onClick?: () => void;
  linkTo?: history.LocationDescriptor;
}

const ReportTab: React.SFC<TabProps> = ({ children, active, onClick, linkTo }) => {
  return (
    <Tab active={active} onClick={onClick} linkTo={linkTo}>
      <Layout padding={{ x: 1 }}>
        <CoreText fontSize={FontSize.Size4} color={active ? Color.Base : Color.Alt2}>
          {children}
        </CoreText>
      </Layout>
    </Tab>
  );
};

export interface PublicProps {
  location?: Location;
}

export interface DataProps {
  shifts?: Shift[];
}

type Props = PublicProps & DataProps;

@withRouter
@observer
export class WorkspaceTabsComponent extends React.Component<Props> {
  private get isShiftLeader(): boolean {
    if (this.props.shifts) {
      return this.props.shifts.some(shift => shift.type === ShiftType.Leader);
    }
    return false;
  }

  public render() {
    const { location } = this.props;
    const { myReports, investigative, allHold } = location
      ? {
          myReports: location.pathname === "/workspaces",
          investigative: location.pathname === "/workspaces/investigative",
          allHold: location.pathname === "/workspaces/all-hold"
        }
      : { myReports: false, investigative: false, allHold: false };
    return (
      <TabWrapper>
        <ReportTab active={myReports} linkTo="/workspaces">
          My Queue
        </ReportTab>
        <ReportTab active={investigative} linkTo="/workspaces/investigative">
          Investigative Queue
        </ReportTab>
        {this.isShiftLeader && (
          <ReportTab active={allHold} linkTo="/workspaces/all-hold">
            All Holds
          </ReportTab>
        )}
      </TabWrapper>
    );
  }
}
