import * as React from "react";

import * as Enzyme from "enzyme";
import * as Adapter from "enzyme-adapter-react-16";

import { Props as ShiftProps } from "aegis/features/shifts/components/shift";
import { Me, Shift, ShiftType } from "aegis/models";
import { MockAdmin, MockMe, MockShift } from "aegis/models/mocks";
import { ActiveShiftsComponent, Props, TestSelectors } from "./component";

Enzyme.configure({ adapter: new Adapter() });

const newActiveShiftsResponse = (shifts: Shift[], meOverrides?: Partial<Me>): Props => {
  return {
    activeShifts: shifts,
    me: MockMe(meOverrides)
  };
};

describe("Active Shifts", () => {
  it("renders with no shifts", () => {
    const props = newActiveShiftsResponse([]);
    const wrapper = Enzyme.shallow(<ActiveShiftsComponent {...props} />);

    expect(wrapper.find({ "data-test-selector": TestSelectors.Shift })).toHaveLength(0);
  });

  it("renders with some other admin", () => {
    const props = newActiveShiftsResponse([MockShift({ admin: MockAdmin({ id: "someone else" }) })]);
    const wrapper = Enzyme.shallow(<ActiveShiftsComponent {...props} />);

    expect(wrapper.find({ "data-test-selector": TestSelectors.Shift })).toHaveLength(1);
  });

  it("renders with multiple shifts", () => {
    const props = newActiveShiftsResponse([MockShift(), MockShift(), MockShift()]);
    const wrapper = Enzyme.shallow(<ActiveShiftsComponent {...props} />);

    expect(wrapper.find({ "data-test-selector": TestSelectors.Shift })).toHaveLength(3);
  });

  it("renders with me", () => {
    const meAdmin = MockAdmin({
      id: "me"
    });
    const props = newActiveShiftsResponse(
      [
        MockShift({
          admin: meAdmin
        })
      ],
      {
        id: "me",
        admin: meAdmin
      }
    );
    const wrapper = Enzyme.shallow(<ActiveShiftsComponent {...props} />);

    expect(wrapper.find({ "data-test-selector": TestSelectors.Shift })).toHaveLength(1);
  });

  it("renders with me as lead with all weight selectors enabled", () => {
    const meAdmin = MockAdmin({
      id: "me"
    });
    const props = newActiveShiftsResponse(
      [
        MockShift(),
        MockShift({
          admin: meAdmin,
          type: ShiftType.Leader
        })
      ],
      {
        id: "me",
        admin: meAdmin
      }
    );
    const wrapper = Enzyme.shallow(<ActiveShiftsComponent {...props} />);
    const shifts = wrapper.find({ "data-test-selector": TestSelectors.Shift });

    expect(shifts).toHaveLength(2);
    shifts.forEach(e => {
      const p: ShiftProps = e.props() as ShiftProps;
      expect(p.enableWeightSelector).toBe(true);
    });
  });
});
