import * as React from "react";

import * as Enzyme from "enzyme";
import * as Adapter from "enzyme-adapter-react-16";

import { MockAdmin, MockShift } from "aegis/models/mocks";

import { ShiftType } from "aegis/models";
import { CheckInButtonComponent, Props, TestSelectors } from "./component";

Enzyme.configure({ adapter: new Adapter() });

const setupProps = (override?: Partial<Props>): Props => {
  return {
    admin: MockAdmin(),
    createShift: jest.fn(),
    error: false,
    loading: false,
    ...override
  };
};

describe("check in button", () => {
  it("renders without error", () => {
    const props = setupProps();
    const wrapper = Enzyme.shallow(<CheckInButtonComponent {...props} />);
    expect(wrapper.find({ "data-test-selector": TestSelectors.Button })).toHaveLength(1);
  });

  it("renders with error", () => {
    const props = setupProps({ error: true });
    const wrapper = Enzyme.shallow(<CheckInButtonComponent {...props} />);
    expect(wrapper.find({ "data-test-selector": TestSelectors.Button })).toHaveLength(1);
  });

  it("renders with loading", () => {
    const props = setupProps({ loading: true });
    const wrapper = Enzyme.shallow(<CheckInButtonComponent {...props} />);
    expect(wrapper.find({ "data-test-selector": TestSelectors.Button })).toHaveLength(1);
  });

  it("does not render when on shift", () => {
    const props = setupProps({
      admin: MockAdmin({
        activeShifts: [MockShift()]
      })
    });
    const wrapper = Enzyme.shallow(<CheckInButtonComponent {...props} />);
    expect(wrapper.find({ "data-test-selector": TestSelectors.Button })).toHaveLength(0);
  });

  it("calls createShift when form submitted", () => {
    const props = setupProps();
    const wrapper = Enzyme.shallow(<CheckInButtonComponent {...props} />);

    const button = wrapper.find({ "data-test-selector": TestSelectors.Button });
    button.simulate("click");

    const form = wrapper.find({ "data-test-selector": TestSelectors.Form });
    expect(form).toHaveLength(1);
    form.prop("onSubmit")(ShiftType.Main, 1);
    expect(props.createShift).toHaveBeenCalledWith(props.admin.id, ShiftType.Main, 1);
  });
});
