import * as React from "react";

import * as Enzyme from "enzyme";
import * as Adapter from "enzyme-adapter-react-16";

import { ShiftType } from "aegis/models";

import { CheckInForm, DEFAULT_WEIGHT, Props, TestSelectors } from "./component";

Enzyme.configure({ adapter: new Adapter() });

const setupProps = (overrides?: Partial<Props>): Props => {
  return {
    onCancel: jest.fn(),
    onSubmit: jest.fn(),
    ...overrides
  };
};

describe("check-in form", () => {
  it("cancels on cancel", () => {
    const props = setupProps();
    const wrapper = Enzyme.shallow(<CheckInForm {...props} />);
    const button = wrapper.find({ "data-test-selector": TestSelectors.CancelButton });
    expect(button).toHaveLength(1);
    button.simulate("click");
    expect(props.onCancel).toHaveBeenCalled();
  });

  it("submits on submit", () => {
    const props = setupProps();
    const wrapper = Enzyme.shallow(<CheckInForm {...props} />);
    const button = wrapper.find({ "data-test-selector": TestSelectors.ConfirmButton });
    expect(button).toHaveLength(1);
    button.simulate("click");
    expect(props.onSubmit).toHaveBeenCalledWith(ShiftType.Main, DEFAULT_WEIGHT);
  });

  it("submits investigative", () => {
    const props = setupProps();
    const wrapper = Enzyme.shallow(<CheckInForm {...props} />);

    const inv = wrapper.find({ "data-test-selector": TestSelectors.InvestigativeTab });
    expect(inv).toHaveLength(1);
    inv.simulate("focus");

    const button = wrapper.find({ "data-test-selector": TestSelectors.ConfirmButton });
    expect(button).toHaveLength(1);
    button.simulate("click");
    expect(props.onSubmit).toHaveBeenCalledWith(ShiftType.Investigative, DEFAULT_WEIGHT);
  });

  it("submits leader with weight", () => {
    const props = setupProps();
    const wrapper = Enzyme.shallow(<CheckInForm {...props} />);

    const inv = wrapper.find({ "data-test-selector": TestSelectors.LeaderTab });
    expect(inv).toHaveLength(1);
    inv.simulate("focus");

    const withWeight = wrapper.find({ "data-test-selector": TestSelectors.LeaderWithWeight });
    const withoutWeight = wrapper.find({ "data-test-selector": TestSelectors.LeaderWithoutWeight });
    expect(withWeight).toHaveLength(1);
    expect(withoutWeight).toHaveLength(1);
    withoutWeight.simulate("focus", {
      currentTarget: {
        value: withoutWeight.prop("value")
      }
    });
    withWeight.simulate("focus", {
      currentTarget: {
        value: withWeight.prop("value")
      }
    });

    const button = wrapper.find({ "data-test-selector": TestSelectors.ConfirmButton });
    expect(button).toHaveLength(1);
    button.simulate("click");
    expect(props.onSubmit).toHaveBeenCalledWith(ShiftType.Leader, DEFAULT_WEIGHT);
  });

  it("submits leader without weight", () => {
    const props = setupProps();
    const wrapper = Enzyme.shallow(<CheckInForm {...props} />);

    const inv = wrapper.find({ "data-test-selector": TestSelectors.LeaderTab });
    expect(inv).toHaveLength(1);
    inv.simulate("focus");

    const withoutWeight = wrapper.find({ "data-test-selector": TestSelectors.LeaderWithoutWeight });
    expect(withoutWeight).toHaveLength(1);
    withoutWeight.simulate("focus", {
      currentTarget: {
        value: withoutWeight.prop("value")
      }
    });

    const button = wrapper.find({ "data-test-selector": TestSelectors.ConfirmButton });
    expect(button).toHaveLength(1);
    button.simulate("click");
    expect(props.onSubmit).toHaveBeenCalledWith(ShiftType.Leader, 0);
  });
});
