import * as React from "react";

import { action, observable } from "mobx";
import { observer } from "mobx-react";

import {
  Button,
  ButtonType,
  CoreText,
  Display,
  FlexDirection,
  JustifyContent,
  Layout,
  Radio,
  SegmentedButton,
  SegmentedButtonOption,
  SegmentedButtonType,
  TextAlign,
  TextType
} from "twitch-core-ui";

import { WeightSelectorComponent } from "aegis/features/shifts/components/shift-weight/components/weight-selector";
import { ShiftType } from "aegis/models";

const CHECK_IN_SL_WEIGHT_RADIO_NAME = "check-in-sl-weight";
const CHECK_IN_TYPE_NAME = "check-in-type";

export const DEFAULT_WEIGHT = 10;

export interface Props {
  onSubmit: (type: ShiftType, weight: number) => void;
  onCancel: () => void;
}

export enum TestSelectors {
  ConfirmButton = "check-in-form--confirm",
  CancelButton = "check-in-form--cancel",
  MainTab = "check-in-form--tab-main",
  InvestigativeTab = "cehck-in-form--tab-investigative",
  LeaderTab = "check-in-form--tab-leader",
  LeaderWithWeight = "check-in-form--tab-leader--with-weight",
  LeaderWithoutWeight = "check-in-form--tab-leader--without-weight"
}

@observer
export class CheckInForm extends React.Component<Props> {
  @observable selectedShift: ShiftType = ShiftType.Main;
  @observable weight = DEFAULT_WEIGHT;

  get shiftOptions(): React.ReactNode {
    const weightInput = <WeightSelectorComponent defaultValue={this.weight} onChange={this.updateWeight} />;

    switch (this.selectedShift) {
      case ShiftType.Main:
        return (
          <Layout display={Display.Flex} flexDirection={FlexDirection.Column} margin={{ bottom: 2 }}>
            <Layout margin={{ bottom: 0.5 }}>
              <CoreText>Set your shift weight:</CoreText>
            </Layout>
            <Layout>
              <Layout display={Display.InlineBlock}>{weightInput}</Layout>
            </Layout>
          </Layout>
        );
      case ShiftType.Leader:
        return (
          <Layout
            display={Display.Flex}
            flexDirection={FlexDirection.Column}
            margin={{ bottom: 2 }}
            textAlign={TextAlign.Left}
          >
            <Layout margin={{ bottom: 1 }}>
              <Layout>
                <Radio
                  name={CHECK_IN_SL_WEIGHT_RADIO_NAME}
                  label="Check in with main queue weight:"
                  value={`${DEFAULT_WEIGHT}`}
                  onFocus={this.updateWeightEvent}
                  data-test-selector={TestSelectors.LeaderWithWeight}
                  defaultChecked
                />
              </Layout>
              <Layout margin={{ top: 0.5 }}>{weightInput}</Layout>
            </Layout>
            <Layout>
              <Radio
                name={CHECK_IN_SL_WEIGHT_RADIO_NAME}
                label="Don't join main queue"
                value="0"
                onFocus={this.updateWeightEvent}
                data-test-selector={TestSelectors.LeaderWithoutWeight}
              />
            </Layout>
          </Layout>
        );
      default:
        return null;
    }
  }

  public render() {
    const { onCancel } = this.props;
    return (
      <Layout
        display={Display.Flex}
        padding={{ x: 5, y: 3 }}
        flexDirection={FlexDirection.Column}
        textAlign={TextAlign.Center}
      >
        <Layout margin={{ bottom: 2 }} fullWidth>
          <CoreText type={TextType.H4}>Shift Check-In</CoreText>
        </Layout>
        <Layout margin={{ bottom: 0.5 }} fullWidth>
          <CoreText>Select a role:</CoreText>
        </Layout>
        <Layout margin={{ bottom: 2 }} fullWidth>
          <SegmentedButton>
            <SegmentedButtonOption
              type={SegmentedButtonType.Radio}
              key={ShiftType.Main}
              label="Main shift"
              name={CHECK_IN_TYPE_NAME}
              onFocus={() => this.selectType(ShiftType.Main)}
              data-test-selector={TestSelectors.MainTab}
              defaultChecked
            />
            <SegmentedButtonOption
              type={SegmentedButtonType.Radio}
              key={ShiftType.Investigative}
              label="Investigative"
              name={CHECK_IN_TYPE_NAME}
              onFocus={() => this.selectType(ShiftType.Investigative)}
              data-test-selector={TestSelectors.InvestigativeTab}
            />
            <SegmentedButtonOption
              type={SegmentedButtonType.Radio}
              key={ShiftType.Leader}
              label="Shift leader"
              name={CHECK_IN_TYPE_NAME}
              onFocus={() => this.selectType(ShiftType.Leader)}
              data-test-selector={TestSelectors.LeaderTab}
            />
          </SegmentedButton>
        </Layout>
        <Layout>{this.shiftOptions}</Layout>
        <Layout display={Display.Flex} justifyContent={JustifyContent.Center} margin={{ top: 0.5 }} fullWidth>
          <Layout margin={{ right: 1 }}>
            <Button
              data-track-click="shift-checkin-form-cancel"
              type={ButtonType.Hollow}
              onClick={onCancel}
              data-test-selector={TestSelectors.CancelButton}
            >
              Cancel
            </Button>
          </Layout>
          <Layout>
            <Button
              data-track-click="shift-checkin-form-submit"
              onClick={this.submit}
              data-test-selector={TestSelectors.ConfirmButton}
            >
              Check in
            </Button>
          </Layout>
        </Layout>
      </Layout>
    );
  }

  @action
  private selectType = (type: ShiftType) => {
    this.selectedShift = type;
  };

  private updateWeightEvent = (e: React.FormEvent<HTMLInputElement>) => {
    this.updateWeight(parseInt(e.currentTarget.value, 10));
  };

  @action
  private updateWeight = (weight: number) => {
    this.weight = weight;
  };

  private submit = () => {
    this.props.onSubmit(this.selectedShift, this.weight);
  };
}
