import * as React from "react";

import gql from "graphql-tag";
import { Mutation, QueryResult } from "react-apollo";

import { Me, PollingInterval, Shift, ShiftType } from "aegis/models";

import { CheckInButtonComponent } from "./component";

import * as Fragments from "aegis/fragments";
import { Query } from "aegis/functionality/apollo-wrapper";
import { FETCH_ACTIVE_SHIFTS } from "../active-shifts-card";

const CREATE_SHIFT = gql`
  mutation createShift($input: CreateShiftInput!) {
    createShift(input: $input) {
      shift {
        ...ShiftFields
      }
    }
  }
  ${Fragments.ShiftFields}
`;

const FETCH_MY_SHIFTS = gql`
  query fetchMyShift {
    me {
      id
      admin {
        id
        activeShifts {
          id
        }
      }
    }
  }
`;

type CreateShiftInput = {
  input: {
    adminID: string;
    type: ShiftType;
    weight: number;
  };
};

type CreateShiftResponse = {
  createShift: {
    shift: Shift;
  };
};

type MyShiftsResponse = {
  me: Me;
};

class CreateShiftMutation extends Mutation<CreateShiftResponse, CreateShiftInput> {}
class MyShiftsQuery extends Query<MyShiftsResponse> {}

export class CheckInButton extends React.Component {
  public render() {
    return (
      <MyShiftsQuery displayName="CheckInButton" query={FETCH_MY_SHIFTS} pollInterval={PollingInterval.Normal}>
        {(meResult: QueryResult<MyShiftsResponse>) => {
          const me: Me | undefined = meResult.data && meResult.data.me;
          const meLoading = meResult.loading && !me;
          if (meResult.error) {
            console.error("check-in query", meResult.error);
          }
          return (
            <CreateShiftMutation mutation={CREATE_SHIFT} refetchQueries={[{ query: FETCH_ACTIVE_SHIFTS }]}>
              {(mutate, { loading, error }) => {
                const onCreate = async (adminID: string, type: ShiftType, weight: number) => {
                  const result = await mutate({
                    variables: { input: { adminID: adminID, type: type, weight: weight } }
                  });
                  if (result) {
                    if (!result.errors) {
                      meResult.refetch();
                    } else {
                      console.error("check-in mutation", result.errors);
                    }
                  }
                };
                if (!me || !me.admin) {
                  return null;
                }
                return (
                  <CheckInButtonComponent
                    admin={me!.admin}
                    createShift={onCreate}
                    error={!!(meResult.error || error)}
                    loading={meLoading || loading}
                  />
                );
              }}
            </CreateShiftMutation>
          );
        }}
      </MyShiftsQuery>
    );
  }
}
