import * as React from "react";

import * as Enzyme from "enzyme";
import * as Adapter from "enzyme-adapter-react-16";

import { Props, TestSelectors, WeightSelectorComponent } from "./component";

Enzyme.configure({ adapter: new Adapter() });

const setupProps = (override?: Partial<Props>): Props => {
  return {
    defaultValue: 10,
    error: false,
    loading: false,
    withSetButton: true,
    onChange: jest.fn(),
    ...override
  };
};

describe("weight selector", () => {
  it("renders without error", () => {
    const props = setupProps();
    const wrapper = Enzyme.shallow(<WeightSelectorComponent {...props} />);
    expect(wrapper.find({ "data-test-selector": TestSelectors.Input })).toHaveLength(1);
    expect(wrapper.find({ "data-test-selector": TestSelectors.Slider })).toHaveLength(1);
    expect(wrapper.find({ "data-test-selector": TestSelectors.ErrorTip })).toHaveLength(0);
    expect(wrapper.find({ "data-test-selector": TestSelectors.SetButton })).toHaveLength(1);
  });

  it("renders without set button", () => {
    const props = setupProps({ withSetButton: false });
    const wrapper = Enzyme.shallow(<WeightSelectorComponent {...props} />);
    expect(wrapper.find({ "data-test-selector": TestSelectors.Input })).toHaveLength(1);
    expect(wrapper.find({ "data-test-selector": TestSelectors.Slider })).toHaveLength(1);
    expect(wrapper.find({ "data-test-selector": TestSelectors.ErrorTip })).toHaveLength(0);
    expect(wrapper.find({ "data-test-selector": TestSelectors.SetButton })).toHaveLength(0);
  });

  it("renders with error", () => {
    const props = setupProps({ error: true });
    const wrapper = Enzyme.shallow(<WeightSelectorComponent {...props} />);
    expect(wrapper.find({ "data-test-selector": TestSelectors.Input })).toHaveLength(1);
    expect(wrapper.find({ "data-test-selector": TestSelectors.Slider })).toHaveLength(1);
    expect(wrapper.find({ "data-test-selector": TestSelectors.ErrorTip })).toHaveLength(1);
    expect(wrapper.find({ "data-test-selector": TestSelectors.SetButton })).toHaveLength(1);
  });

  it("submits when on bluring slider", () => {
    const props = setupProps();
    const wrapper = Enzyme.shallow(<WeightSelectorComponent {...props} />);
    const input = wrapper.find({ "data-test-selector": TestSelectors.Input });
    expect(input).toHaveLength(1);
    input.simulate("blur");
    expect(props.onChange).toHaveBeenCalledWith(props.defaultValue);
  });

  it("submits when on clicking set", () => {
    const props = setupProps();
    const wrapper = Enzyme.shallow(<WeightSelectorComponent {...props} />);
    const input = wrapper.find({ "data-test-selector": TestSelectors.SetButton });
    expect(input).toHaveLength(1);
    input.simulate("click");
    expect(props.onChange).toHaveBeenCalledWith(props.defaultValue);
  });
});
