import * as React from "react";

import { action, observable } from "mobx";
import { observer } from "mobx-react";

import { Button, Display, FlexDirection, Input, InputType, Layout, Range, Tooltip } from "twitch-core-ui";

const MIN_WEIGHT = 0;
const MAX_WEIGHT = 100;

export interface Props {
  defaultValue: number;
  error?: boolean;
  loading?: boolean;
  withSetButton?: boolean;
  onChange: (weight: number) => void;
}

export enum TestSelectors {
  Slider = "shift-weight-selector--slider",
  Input = "shift-weight-selector--input",
  SetButton = "shift-weight-selector--set-button",
  ErrorTip = "shift-weight-selector--error-tooltip"
}

@observer
export class WeightSelectorComponent extends React.Component<Props> {
  @observable private weight = this.props.defaultValue;

  public render() {
    const { error, loading, withSetButton } = this.props;
    const inputs = (
      <Layout display={Display.Flex} flexDirection={FlexDirection.Row}>
        <Layout className="shift-weight-selector-input">
          <Input
            type={InputType.Number}
            error={!!error}
            disabled={!!loading}
            min={0}
            max={100}
            step={1}
            value={this.weight.toString()}
            onChange={this.setSelectedWeight}
            onBlur={this.submitSelectedWeight}
            data-test-selector={TestSelectors.Input}
          />
        </Layout>
        <Layout padding={{ x: 1 }}>
          <Range
            error={!!error}
            disabled={!!loading}
            min={0}
            max={100}
            step={1}
            value={this.weight.toString()}
            onChange={this.setSelectedWeight}
            onBlur={this.submitSelectedWeight}
            data-test-selector={TestSelectors.Slider}
          />
        </Layout>
        {withSetButton && (
          <Button
            data-track-click="shift-weight-set"
            onClick={this.submitSelectedWeight}
            data-test-selector={TestSelectors.SetButton}
          >
            Set
          </Button>
        )}
      </Layout>
    );

    if (error) {
      return (
        <Tooltip
          label="An error occurred when updating the weight. Please try again."
          data-test-selector={TestSelectors.ErrorTip}
        >
          {inputs}
        </Tooltip>
      );
    }

    return inputs;
  }

  @action
  private submitSelectedWeight = () => {
    if (this.weight < MIN_WEIGHT) {
      this.weight = MIN_WEIGHT;
    } else if (this.weight > MAX_WEIGHT) {
      this.weight = MAX_WEIGHT;
    }
    this.props.onChange(this.weight);
  };

  @action
  private setSelectedWeight = (e: React.KeyboardEvent<HTMLInputElement>) => {
    this.weight = e.currentTarget.valueAsNumber;
  };
}
