import * as React from "react";

import gql from "graphql-tag";
import { Mutation } from "react-apollo";

import * as Fragments from "aegis/fragments";
import { Shift } from "aegis/models";

import { errorNotification } from "aegis/functionality/error-notifications";
import { apolloErrorHandler } from "aegis/functionality/error-notifications";
import { WeightSelectorComponent } from "./component";
import "./styles.scss";

export const modifyShift = gql`
  mutation modifyShift($input: ModifyShiftInput!) {
    modifyShift(input: $input) {
      shift {
        ...ShiftFields
      }
    }
  }
  ${Fragments.ShiftFields}
`;

type ModifyShiftInput = {
  input: {
    id: string;
    weight: number;
  };
};

type ModifyShiftResponse = {
  modifyShift: {
    shift: Shift;
  };
};

export interface WeightSelectorProps {
  defaultValue: number;
  shiftID: string;
  withSetButton?: boolean;
  onChange?: () => void;
}

export class ModifyShiftMutation extends Mutation<ModifyShiftResponse, ModifyShiftInput> {}

export class WeightSelector extends React.Component<WeightSelectorProps> {
  public render() {
    const { shiftID, defaultValue, withSetButton } = this.props;
    return (
      <ModifyShiftMutation
        mutation={modifyShift}
        onCompleted={() => {
          if (this.props.onChange) {
            this.props.onChange();
          }
        }}
        onError={apolloErrorHandler("Failed to delete routes")}
      >
        {(mutate, { loading, error }) => {
          const onUpdate = async (w: number) => {
            const result = await mutate({
              variables: { input: { id: shiftID, weight: w } }
            });
            if (result && result.errors) {
              errorNotification(result.errors.toString());
            }
          };
          if (error) {
            errorNotification(error.message);
          }
          return (
            <WeightSelectorComponent
              withSetButton={withSetButton}
              error={!!error}
              loading={loading}
              onChange={onUpdate}
              defaultValue={defaultValue}
            />
          );
        }}
      </ModifyShiftMutation>
    );
  }
}
