import * as React from "react";

import * as FontAwesome from "react-fontawesome";

import { ShiftWeight } from "aegis/features/shifts";
import { Shift as ShiftModel, ShiftType } from "aegis/models";
import { CheckOutButton } from "../check-out-button";

import {
  AlignItems,
  Background,
  CoreImage,
  CoreText,
  Display,
  FlexDirection,
  InjectLayout,
  JustifyContent,
  Layout,
  Overflow,
  Position,
  StyledLayout,
  TextAlign,
  TextTransform,
  TextType
} from "twitch-core-ui";

import "./styles.scss";

export interface Props {
  shift: ShiftModel;
  enableWeightSelector: boolean;
  isMyShift?: boolean;
}

export enum TestSelectors {
  ShiftWeight = "shift-shift--weight"
}

export class Shift extends React.Component<Props> {
  get weightSelector(): JSX.Element | null {
    const { shift, enableWeightSelector } = this.props;
    switch (shift.type) {
      case ShiftType.Investigative:
        return null;
      default:
        return (
          <ShiftWeight shift={shift} enabled={enableWeightSelector} data-test-selector={TestSelectors.ShiftWeight} />
        );
    }
  }

  get icon(): JSX.Element | null {
    const { shift } = this.props;
    switch (shift.type) {
      case ShiftType.Investigative:
        return <FontAwesome name="search" />;
      case ShiftType.Leader:
        return <FontAwesome name="users" />;
      case ShiftType.Main:
        return <FontAwesome name="star" />;
      default:
        console.error("Unexpected shift type, no icon:", shift.type);
        return null;
    }
  }

  public render() {
    const { shift, isMyShift } = this.props;
    return (
      <StyledLayout borderBottom overflow={Overflow.Visible}>
        <Layout
          display={Display.Flex}
          alignItems={AlignItems.Center}
          justifyContent={JustifyContent.Between}
          overflow={Overflow.Visible}
          position={Position.Relative}
          textAlign={TextAlign.Center}
          className="shift--container"
          fullWidth
        >
          <Layout>
            <InjectLayout margin={1}>
              <CoreImage alt="Avatar" src={shift.admin.avatarURL} className="shift-avatar" />
            </InjectLayout>
          </Layout>
          <Layout display={Display.Flex} flexDirection={FlexDirection.Column} textAlign={TextAlign.Left} flexGrow={2}>
            <CoreText bold>{(shift.admin && shift.admin.ldap) || "<unknown>"}</CoreText>
            {this.weightSelector}
          </Layout>
          <StyledLayout background={Background.Overlay} margin={{ right: 1 }} className="shift--type-container">
            <StyledLayout textAlign={TextAlign.Center} className="shift--type-part">
              <Layout display={Display.InlineBlock} margin={{ x: 0.5 }}>
                {this.icon}
              </Layout>
              <CoreText transform={TextTransform.Capitalize} type={TextType.Span}>
                {shift.type}
              </CoreText>
            </StyledLayout>
          </StyledLayout>
          <StyledLayout
            background={Background.Alt}
            display={Display.Flex}
            justifyContent={JustifyContent.Center}
            alignItems={AlignItems.Center}
            position={Position.Absolute}
            attachRight
            overflow={Overflow.Hidden}
            fullHeight
            className="shift--checkout"
          >
            <CheckOutButton shift={shift} confirm={!isMyShift} />
          </StyledLayout>
        </Layout>
      </StyledLayout>
    );
  }
}
