import * as React from "react";
import { NavLink } from "react-router-dom";

import { Me } from "aegis/features/me";
import {
  AlignItems,
  Background,
  Display,
  FlexDirection,
  FlexWrap,
  InjectLayout,
  Interactable,
  Layout,
  LayoutProps,
  Position,
  StyledLayout,
  SVG,
  SVGAsset
} from "twitch-core-ui";
import { TopNavDropdown } from "./components/dropdown-menu";

import "./styles.scss";

export enum TestSelectors {
  TopNavAccountOverview = "top-nav__account-overview",
  TopNavAltLookup = "top-nav__alt-lookup",
  TopNavBanForm = "top-nav__ban-form",
  TopNavBounties = "top-nav__bounties",
  TopNavChatActivity = "top-nav__chat-activity",
  TopNavChatlog = "top-nav__chatlog",
  TopNavClipMassDeleter = "top-nav__clip-mass-deleter",
  TopNavEnforcementSearch = "top-nav__enforcement-search",
  TopNavEmoticons = "top-nav__emoticons",
  TopNavImageUploader = "top-nav__image-uploader",
  ReportSearch = "top-nav__report-search",
  GameDirectory = "top-nav__game-directory"
}

export class TopNav extends React.Component<{}, {}> {
  public render() {
    const NavItemProps: LayoutProps = {
      position: Position.Relative,
      display: Display.Flex,
      flexGrow: 0,
      alignItems: AlignItems.Stretch,
      fullHeight: true
    };

    const tools: { linkTo: string; name: string; testSelector: TestSelectors }[] = [
      { linkTo: "/accountinfo", name: "Account Overview", testSelector: TestSelectors.TopNavAccountOverview },
      { linkTo: "/altlookup", name: "Alt lookup", testSelector: TestSelectors.TopNavAltLookup },
      { linkTo: "/ban", name: "Ban Form", testSelector: TestSelectors.TopNavBanForm },
      { linkTo: "/bounties", name: "Bounties", testSelector: TestSelectors.TopNavBounties },
      { linkTo: "/chatactivity", name: "Chat Activity", testSelector: TestSelectors.TopNavChatActivity },
      { linkTo: "/chatlogs", name: "Chat Logs", testSelector: TestSelectors.TopNavChatlog },
      { linkTo: "/clips/deleter", name: "Clips Mass Deleter", testSelector: TestSelectors.TopNavClipMassDeleter },
      { linkTo: "/enforcementsearch", name: "Enforcement Search", testSelector: TestSelectors.TopNavEnforcementSearch },
      { linkTo: "/emoticons", name: "Emoticons", testSelector: TestSelectors.TopNavEmoticons },
      { linkTo: "/imageupload", name: "Image Uploader", testSelector: TestSelectors.TopNavImageUploader },
      { linkTo: "/reportsearch", name: "Report Search", testSelector: TestSelectors.ReportSearch },
      { linkTo: "/gamedirectory", name: "Game Directory", testSelector: TestSelectors.GameDirectory }
    ];

    const menuItems = [
      <Layout key="navitem-workspaces" {...NavItemProps}>
        <Interactable data-track-click="top-nav-workspaces" linkTo="/workspaces">
          <Layout className="top-nav__nav-item" display={Display.Flex} fullHeight>
            Workspaces
          </Layout>
        </Interactable>
      </Layout>,
      <Layout key="navitem-tools" {...NavItemProps}>
        <TopNavDropdown
          topnavElement={
            <Layout display={Display.Flex} className="top-nav__nav-item" flexWrap={FlexWrap.NoWrap} fullHeight>
              <Layout flexGrow={1}>Tools</Layout>
              <StyledLayout
                display={Display.Flex}
                flexShrink={0}
                padding={{ left: 0.5 }}
                data-a-target="top-nav-dropdown-arrow"
              >
                <SVG asset={SVGAsset.GlyphArrDown} />
              </StyledLayout>
            </Layout>
          }
          dropdownElement={
            <StyledLayout
              padding={{ y: 1 }}
              display={Display.Flex}
              flexDirection={FlexDirection.Column}
              background={Background.Alt}
            >
              {tools.map(function(item: { linkTo: string; name: string; testSelector: string }, i: number) {
                return (
                  <Interactable
                    data-track-click={`top-nav-tool-${item.name}`}
                    linkTo={item.linkTo}
                    key={`topnav-tool-${i}`}
                    data-test-selector={item.testSelector}
                  >
                    <StyledLayout className="top-nav__dropdown-item" padding={{ x: 2, y: 0.5 }}>
                      {item.name}
                    </StyledLayout>
                  </Interactable>
                );
              })}
            </StyledLayout>
          }
          display={Display.Flex}
          closeOnClick
        />
      </Layout>,
      <Layout key="navitem-records" {...NavItemProps}>
        <Interactable data-track-click="top-nav-records" linkTo="/records">
          <Layout className="top-nav__nav-item" display={Display.Flex} fullHeight>
            Records
          </Layout>
        </Interactable>
      </Layout>
    ];

    return (
      <InjectLayout position={Position.Fixed} attachTop fullWidth>
        <nav className="top-nav" data-a-target="top-nav-container">
          <Layout
            className="top-nav__menu"
            display={Display.Flex}
            alignItems={AlignItems.Stretch}
            flexWrap={FlexWrap.NoWrap}
            fullHeight
          >
            <InjectLayout key="topnavLogo" display={Display.InlineFlex} alignItems={AlignItems.Center} flexShrink={0}>
              <NavLink to="/" className="top-nav__home-link" data-a-target="home-link">
                <SVG asset={SVGAsset.Automod} width={32} height={32} />
              </NavLink>
            </InjectLayout>
            <Layout
              className="top-nav__nav-items-container"
              display={Display.Flex}
              alignItems={AlignItems.Center}
              flexGrow={1}
              flexWrap={FlexWrap.NoWrap}
              key="topnavItems"
            >
              {menuItems}
            </Layout>
            <Layout
              className="top-nav__nav-items-container"
              display={Display.Flex}
              alignItems={AlignItems.Center}
              flexGrow={0}
              flexWrap={FlexWrap.NoWrap}
              key="topnavAccountItems"
            >
              <Me />
            </Layout>
          </Layout>
        </nav>
      </InjectLayout>
    );
  }
}
