import * as React from "react";

import { inject } from "mobx-react";

import { AncestorName, AncestorNameDataAttribute } from "aegis/context/ancestor";
import { TrackingClientProps } from "./client";
import { Click, EventName } from "./models";

const trackedClickDataAttribute = "data-track-click";
const trackedReportDataAttribute = "data-track-report";

type Props = TrackingClientProps;

// Click tracker that tracks all click events.
// Click event will be tracked ONLY IF one of the parent have "data-track-click" attribute;
// event name will be the value of the data attribute.
@inject("trackingClient")
export class ClickTracker extends React.Component<Props> {
  public componentDidMount() {
    document.addEventListener("click", this.mouseClicked, true);
  }

  public componentWillUnmount() {
    document.removeEventListener("click", this.mouseClicked, true);
  }

  public render() {
    return null;
  }

  private mouseClicked = (e: MouseEvent) => {
    const target = e.target as HTMLElement;

    // Workaround for https://stackoverflow.com/questions/24501497/why-the-onclick-element-will-trigger-twice-for-label-element
    // Some core-ui element wrap input inside the label
    if (target.parentElement && target.parentElement.tagName === "LABEL") {
      if (target.tagName !== "INPUT") {
        return;
      }
    }

    const trackedNode = target.closest(`[${trackedClickDataAttribute}]`);
    if (trackedNode) {
      const ancestorNameNode = trackedNode.closest(`[${AncestorNameDataAttribute}]`);
      const reportNode = trackedNode.closest(`[${trackedReportDataAttribute}]`);

      const event: Click = {
        action: trackedNode.getAttribute(trackedClickDataAttribute)!,
        ui_context: ancestorNameNode ? ancestorNameNode.getAttribute(AncestorNameDataAttribute)! : AncestorName.Root,
        report_id: reportNode && reportNode.getAttribute(trackedReportDataAttribute)
      };

      this.props.trackingClient!.Track(EventName.Click, event);
    }
  };
}
