import { MockAdmin } from "aegis/models/mocks";
import { AuthStore } from "aegis/stores";
import { ContextualEvent, EventName } from "../models";
import { AWAIT_LOGIN_INTERVAL, TrackingClient } from "./component";
import { MockReporter } from "./mock";

describe("tracking client", () => {
  beforeAll(() => {
    jest.useFakeTimers();
  });

  afterAll(() => {
    jest.useRealTimers();
  });

  it("tracks event when admin is logged in", () => {
    const admin = MockAdmin();
    const authStore = new AuthStore();
    authStore.userLogin(admin);
    const reporter = new MockReporter();
    const client = new TrackingClient(reporter, authStore);

    const event1: ContextualEvent = {
      ui_context: "context1"
    };

    const event2: ContextualEvent = {
      ui_context: "context2"
    };

    client.Track(EventName.Click, event1);
    client.Track(EventName.Heartbeat, event2);

    const sentEvents = reporter.getReportedEvents();
    expect(sentEvents).toHaveLength(2);
    expect(sentEvents[0].event).toBe(EventName.Click);
    expect(sentEvents[0].properties).toHaveProperty("ui_context", event1.ui_context);
    expect(sentEvents[0].properties).toHaveProperty("admin_id", admin.id);

    expect(sentEvents[1].event).toBe(EventName.Heartbeat);
    expect(sentEvents[1].properties).toHaveProperty("ui_context", event2.ui_context);
    expect(sentEvents[1].properties).toHaveProperty("admin_id", admin.id);
  });

  it("tracks event when admin is not logged in", () => {
    const authStore = new AuthStore();
    const reporter = new MockReporter();
    const client = new TrackingClient(reporter, authStore);

    const event1: ContextualEvent = {
      ui_context: "context1"
    };

    const event2: ContextualEvent = {
      ui_context: "context2"
    };

    client.Track(EventName.Click, event1);
    client.Track(EventName.Heartbeat, event2);

    jest.runAllTimers();

    const sentEvents = reporter.getReportedEvents();
    expect(sentEvents).toHaveLength(2);
    expect(sentEvents[0].event).toBe(EventName.Click);
    expect(sentEvents[0].properties).toHaveProperty("ui_context", event1.ui_context);
    expect(sentEvents[0].properties).toHaveProperty("admin_id", undefined);

    expect(sentEvents[1].event).toBe(EventName.Heartbeat);
    expect(sentEvents[1].properties).toHaveProperty("ui_context", event2.ui_context);
    expect(sentEvents[1].properties).toHaveProperty("admin_id", undefined);
  });

  it("tracks event when admin login information is loaded after event is sent", () => {
    const admin = MockAdmin();
    const authStore = new AuthStore();
    const reporter = new MockReporter();
    const client = new TrackingClient(reporter, authStore);

    const event1: ContextualEvent = {
      ui_context: "context1"
    };

    const event2: ContextualEvent = {
      ui_context: "context2"
    };

    client.Track(EventName.Click, event1);
    client.Track(EventName.Heartbeat, event2);

    expect(reporter.getReportedEvents()).toHaveLength(0); // Events shouldn't be sent yet
    authStore.userLogin(admin);
    jest.advanceTimersByTime(AWAIT_LOGIN_INTERVAL);

    const sentEvents = reporter.getReportedEvents();
    expect(sentEvents).toHaveLength(2);
    expect(sentEvents[0].event).toBe(EventName.Click);
    expect(sentEvents[0].properties).toHaveProperty("ui_context", event1.ui_context);
    expect(sentEvents[0].properties).toHaveProperty("admin_id", admin.id);

    expect(sentEvents[1].event).toBe(EventName.Heartbeat);
    expect(sentEvents[1].properties).toHaveProperty("ui_context", event2.ui_context);
    expect(sentEvents[1].properties).toHaveProperty("admin_id", admin.id);
  });
});
