import { Reporter, SpadeEvent } from "tachyon-spade-reporter";

import { AuthStore } from "aegis/stores";
import { BaseEvent, EventName } from "../models";

export interface TrackingClientProps {
  trackingClient?: TrackingClient;
}

// Following vars are exported for test
export const MAX_RETRY_ATTEMPTS = 5; // attempts to retry fetching logged in user infromation
export const AWAIT_LOGIN_INTERVAL = 1000; // ms to wait before checking if user is logged in

export class TrackingClient {
  private trackingClient: Reporter<SpadeEvent>;
  private authStore: AuthStore;

  constructor(reporter: Reporter<SpadeEvent>, authStore: AuthStore) {
    this.trackingClient = reporter;
    this.authStore = authStore;
  }

  public Track(name: EventName, event: BaseEvent) {
    this.TrackEventInternal(name, event, 0);
  }

  private TrackEventInternal = (name: EventName, event: BaseEvent, attempt: number) => {
    // Handle the case where an event was sent too fast before
    // the page could finish loading user login information
    if (!this.authStore.loggedInUser) {
      if (attempt < MAX_RETRY_ATTEMPTS) {
        // Try again after a second
        setTimeout(() => this.TrackEventInternal(name, event, attempt + 1), AWAIT_LOGIN_INTERVAL);
        return;
      }
      // Otherwise go ahead and send the event without user login information
    }
    const props: BaseEvent = {
      ...event,
      window_location: location.pathname,
      host: location.hostname,
      admin_id: this.authStore.loggedInUser ? this.authStore.loggedInUser.id : undefined
    };

    this.trackingClient.reportEvent({
      event: name,
      properties: props
    });
  };
}
