import * as React from "react";

import { inject } from "mobx-react";

import { ReportStoreProps } from "aegis/stores";
import { TrackingClientProps } from "./client";
import { EventName, Heartbeat, MemoryEventData } from "./models";

type Props = TrackingClientProps & ReportStoreProps;

@inject("trackingClient")
@inject("reportStore")
export class HeartbeatTracker extends React.Component<Props> {
  private isTabInFocus: boolean = true;
  private mouseClicked: number = 0;
  private metricsIntervalID: NodeJS.Timer;

  public componentDidMount() {
    document.addEventListener("click", this.mouseClickedTracker, true);

    // We need to update this code to handle multiple window.onblur/onfocus hook if this were to happen
    if (window.onblur || window.onfocus) {
      console.error("Window.onfocus/onblur is already defined. Overriding...");
    }
    window.onblur = this.onWindowBlur;
    window.onfocus = this.onWindowFocus;

    this.metricsIntervalID = setInterval(this.sendMetrics, 5000);
  }

  public componentWillUnmount() {
    document.removeEventListener("click", this.mouseClickedTracker, true);

    window.onblur = null;
    window.onfocus = null;

    clearInterval(this.metricsIntervalID);
  }

  public render() {
    return null;
  }

  private onWindowBlur = () => {
    this.isTabInFocus = false;
  };

  private onWindowFocus = () => {
    this.isTabInFocus = true;
  };

  private mouseClickedTracker = () => {
    this.mouseClicked++;
  };

  private get memoryUsageData(): MemoryEventData | undefined {
    // Check if performance.memory exists (Chrome exclusive)
    if (!window.performance.memory) {
      return;
    }
    return {
      js_total_heap_size_bytes: window.performance.memory.totalJSHeapSize,
      js_used_heap_size_bytes: window.performance.memory.usedJSHeapSize,
      js_heap_size_limit_bytes: window.performance.memory.jsHeapSizeLimit
    };
  }

  private sendMetrics = () => {
    const { trackingClient, reportStore } = this.props;

    const event: Heartbeat = {
      tab_in_focus: this.isTabInFocus,
      page_visibility: !document.hidden, // HTML5 visibility api
      mouse_clicks: this.mouseClicked,
      ...this.memoryUsageData
    };

    if (
      reportStore &&
      (reportStore.expandedIDs.size > 0 || reportStore.visibleReportsIDs.size > 0 || reportStore.hasFocusedID)
    ) {
      let formattedExpandedIDs = "_";
      reportStore.expandedIDs.forEach(id => {
        formattedExpandedIDs = formattedExpandedIDs + id + "_";
      });
      event.reports_in_view = reportStore.visibleReportsIDs.size;
      event.focused_report_id = reportStore.focusedID;
      event.expanded_report_ids = formattedExpandedIDs;
    }

    trackingClient!.Track(EventName.Heartbeat, event);

    // Reset mouse click after event is sent
    this.mouseClicked = 0;
  };
}
