import * as React from "react";

import gql from "graphql-tag";

import { Query } from "aegis/functionality/apollo-wrapper";
import { PollingInterval } from "aegis/models";
import { Color, CoreText, LoadingSpinner } from "twitch-core-ui";
import { UnresolvedReportsComponent } from "./component";

export const FETCH_UNRESOLVED_REPORT_COUNTS = gql`
  query unresolvedReportCounts {
    unresolvedReportCounts {
      total
      mainQueueRouted
      investigativeQueue
      mainQueueAssigned
    }
  }
`;

export type UnresolvedReportCountsResponse = {
  unresolvedReportCounts: {
    total: number;
    mainQueueRouted: number;
    investigativeQueue: number;
    mainQueueAssigned: number;
  };
};

export class UnresolvedReportCountQuery extends Query<UnresolvedReportCountsResponse> {}

export const UnresolvedReportsCard = () => (
  <UnresolvedReportCountQuery
    displayName="UnresolvedReportsCard"
    pollInterval={PollingInterval.Normal}
    query={FETCH_UNRESOLVED_REPORT_COUNTS}
  >
    {({ loading, error, data }) => {
      if (error) {
        console.error(error);
        return (
          <CoreText color={Color.Error}>An error occurred while loading unresolved reports: {error.message}</CoreText>
        );
      }

      if (loading && !data) {
        return <LoadingSpinner />;
      }

      if (!data || !data.unresolvedReportCounts) {
        return null;
      } else {
        return (
          <UnresolvedReportsComponent
            total={data.unresolvedReportCounts.total}
            mainQueueRouted={data.unresolvedReportCounts.mainQueueRouted}
            investigativeQueue={data.unresolvedReportCounts.investigativeQueue}
            mainQueueAssigned={data.unresolvedReportCounts.mainQueueAssigned}
          />
        );
      }
    }}
  </UnresolvedReportCountQuery>
);
