import * as React from "react";

import * as Enzyme from "enzyme";
import * as Adapter from "enzyme-adapter-react-16";

import { shallow } from "enzyme";
import { ChannelStatusIndicatorStatus } from "twitch-core-ui";

import { DetailedUser } from "aegis/models";
import { MockDetailedUser } from "aegis/models/mocks";
import { Props, TestSelector, UserCardComponent } from "./component";

import "aegis/functionality/date-extensions";

Enzyme.configure({ adapter: new Adapter() });

const setup = (overrides?: Partial<DetailedUser>, loading?: boolean) => {
  const combinedProps: Props = {
    user: MockDetailedUser(overrides),
    loading: !!loading
  };
  return {
    props: combinedProps,
    wrapper: shallow(<UserCardComponent {...combinedProps} />)
  };
};

describe("User Card", () => {
  it("Renders while loading state for the user card", () => {
    const { wrapper } = setup(undefined, true);

    expect(wrapper.find({ "data-test-selector": TestSelector.LIVE_INDICATOR_SELECTOR }).props()).toMatchObject({
      status: ChannelStatusIndicatorStatus.Live
    });

    expect(
      wrapper.find({ "data-test-selector": TestSelector.CHANNEL_PAGE_BUTTON_SELECTOR }).prop("disabled")
    ).toBeTruthy();
    expect(wrapper.find({ "data-test-selector": TestSelector.CHATLOGS_BUTTON_SELECTOR }).prop("disabled")).toBeTruthy();
    expect(
      wrapper.find({ "data-test-selector": TestSelector.ADMIN_PANEL_BUTTON_SELECTOR }).prop("disabled")
    ).toBeTruthy();
  });

  it("Renders a default view of the user card", () => {
    const { wrapper, props } = setup();

    expect(wrapper.find({ "data-test-selector": TestSelector.USER_ROLE_TAGS_SELECTOR }).props()).toMatchObject({
      roles: props.user!.roles,
      expanded: true
    });

    expect(wrapper.find({ "data-test-selector": TestSelector.ACCOUNT_CREATED_TEXT_SELECTOR }).props().children).toEqual(
      "Account created 2015-03-07"
    );

    expect(wrapper.find({ "data-test-selector": TestSelector.LIVE_INDICATOR_TEXT_SELECTOR }).props().children).toEqual(
      "live now"
    );

    expect(wrapper.find({ "data-test-selector": TestSelector.LIVE_INDICATOR_SELECTOR }).props()).toMatchObject({
      status: ChannelStatusIndicatorStatus.Live
    });

    expect(wrapper.find({ "data-test-selector": TestSelector.ACCOUNT_INFO_BUTTON_SELECTOR }).props().children).toEqual(
      "Account Overview"
    );

    expect(wrapper.find({ "data-test-selector": TestSelector.CHATLOGS_BUTTON_SELECTOR }).props().children).toEqual(
      "Chatlogs"
    );
    expect(wrapper.find({ "data-test-selector": TestSelector.CHATLOGS_BUTTON_SELECTOR }).prop("disabled")).toBeFalsy();

    expect(wrapper.find({ "data-test-selector": TestSelector.ADMIN_PANEL_BUTTON_SELECTOR }).props().children).toEqual(
      "Admin Panel"
    );
    expect(
      wrapper.find({ "data-test-selector": TestSelector.ADMIN_PANEL_BUTTON_SELECTOR }).prop("disabled")
    ).toBeFalsy();

    expect(wrapper.find({ "data-test-selector": TestSelector.CHANNEL_PAGE_BUTTON_SELECTOR }).props().children).toEqual(
      "Channel Page"
    );
    expect(
      wrapper.find({ "data-test-selector": TestSelector.CHANNEL_PAGE_BUTTON_SELECTOR }).prop("disabled")
    ).toBeFalsy();

    expect(wrapper.find({ "data-test-selector": TestSelector.VOD_MANAGER_BUTTON_SELECTOR }).props().children).toEqual(
      "Vod Manager"
    );
    expect(
      wrapper.find({ "data-test-selector": TestSelector.VOD_MANAGER_BUTTON_SELECTOR }).prop("disabled")
    ).toBeFalsy();
  });

  it("Renders the correct text when the user account creation date is unknown", () => {
    const { wrapper } = setup({ createdAt: undefined });

    expect(wrapper.find({ "data-test-selector": TestSelector.ACCOUNT_CREATED_TEXT_SELECTOR }).props().children).toEqual(
      "Unknown account creation date"
    );
  });

  it("Renders the correct channel live indicator status and text when the user is not streaming", () => {
    const { wrapper } = setup({ isLiveStreaming: false });

    expect(wrapper.find({ "data-test-selector": TestSelector.LIVE_INDICATOR_TEXT_SELECTOR }).props().children).toEqual(
      "not live"
    );

    expect(wrapper.find({ "data-test-selector": TestSelector.LIVE_INDICATOR_SELECTOR }).props()).toMatchObject({
      status: ChannelStatusIndicatorStatus.Offline
    });
  });

  it("Opens a new window when clicking the See Channel Page button", () => {
    const { wrapper } = setup();

    expect(wrapper.find({ "data-test-selector": TestSelector.CHANNEL_PAGE_BUTTON_SELECTOR }).props().linkTo).toBe(
      "https://www.twitch.tv/login?tmi_host=false"
    );
    expect(wrapper.find({ "data-test-selector": TestSelector.CHANNEL_PAGE_BUTTON_SELECTOR }).props().targetBlank).toBe(
      true
    );
  });

  it("Renders the chatlogs button with a proper link to chatlogs page", () => {
    const { wrapper } = setup();

    expect(wrapper.find({ "data-test-selector": TestSelector.CHATLOGS_BUTTON_SELECTOR }).props().linkTo).toBe(
      "/chatlogs?userids=id"
    );
    expect(wrapper.find({ "data-test-selector": TestSelector.CHATLOGS_BUTTON_SELECTOR }).props().targetBlank).toBe(
      true
    );
  });
});
