import * as React from "react";

import { AncestorName, AncestorNameContextProvider } from "aegis/context/ancestor";
import { UserRoleTags } from "aegis/features/user-role-tags";
import { IndicatorButton } from "aegis/functionality/components/indicator-button/component";
import { accountInfoLink, channelLink, DetailedUser } from "aegis/models";
import { UserCardHeader } from "./components/user-card-header";
import { UserCardSuspensionDetails } from "./components/user-card-suspension-details";

import {
  AlignItems,
  ButtonSize,
  ButtonType,
  Card,
  ChannelStatusIndicator,
  ChannelStatusIndicatorStatus,
  Color,
  CoreText,
  Display,
  FlexDirection,
  FontSize,
  JustifyContent,
  Layout,
  Placeholder,
  StyledLayout,
  SVGAsset,
  TextAlign,
  TextType
} from "twitch-core-ui";

import { OnEscalateList } from "../user/components/on-escalate-list";
import "./style.scss";

export type Props = {
  loading: boolean;
  user: DetailedUser;
};

export enum TestSelector {
  ACCOUNT_CREATED_TEXT_SELECTOR = "account-created-text-selector",
  ACCOUNT_INFO_BUTTON_SELECTOR = "account-info-button-selector",
  CHANNEL_PAGE_BUTTON_SELECTOR = "channel-page-button-selector",
  VOD_MANAGER_BUTTON_SELECTOR = "vod-manager-button-selector",
  ADMIN_PANEL_BUTTON_SELECTOR = "admin-panel-button-selector",
  CHATLOGS_BUTTON_SELECTOR = "chatlogs-button-selector",
  LIVE_INDICATOR_SELECTOR = "live-indicator-selector",
  LIVE_INDICATOR_TEXT_SELECTOR = "live-indicator-text-selector",
  USER_ROLE_TAGS_SELECTOR = "user-role-tags-selector"
}

export class UserCardComponent extends React.Component<Props> {
  public render() {
    const { user, loading } = this.props;

    let userRoles = <Placeholder data-test-selector={TestSelector.USER_ROLE_TAGS_SELECTOR} />;
    let accountCreatedText = <Placeholder data-test-selector={TestSelector.ACCOUNT_CREATED_TEXT_SELECTOR} />;
    let liveIndicatorText = <Placeholder data-test-selector={TestSelector.LIVE_INDICATOR_TEXT_SELECTOR} />;
    let liveIndicator = ChannelStatusIndicatorStatus.Offline;
    const chatlogsLink = `/chatlogs?userids=${user.id}`;
    const userProfileLink = channelLink(user);
    const adminPanelLink = `https://admin-panel.internal.justin.tv/twitch/users/${user.id}`;
    const vodManagerLink = `https://www.twitch.tv/${user.login}/manager/past_broadcasts?tmi_host=false`;

    userRoles = (
      <Layout display={Display.Flex} flexDirection={FlexDirection.Row} justifyContent={JustifyContent.Start}>
        <OnEscalateList IsEscalatePartner={user.isEscalatePartner} />
        <UserRoleTags roles={user.roles} expanded data-test-selector={TestSelector.USER_ROLE_TAGS_SELECTOR} />
      </Layout>
    );
    if (user.createdAt) {
      const timestamp = Date.parse(user.createdAt);
      const createdAt = new Date(timestamp);
      accountCreatedText = (
        <CoreText data-test-selector={TestSelector.ACCOUNT_CREATED_TEXT_SELECTOR} type={TextType.Span}>
          {`Account created ${createdAt.toDisplayFormat().substring(0, 10)}`}
        </CoreText>
      );
    } else {
      accountCreatedText = (
        <CoreText data-test-selector={TestSelector.ACCOUNT_CREATED_TEXT_SELECTOR} type={TextType.Span}>
          {"Unknown account creation date"}
        </CoreText>
      );
    }

    if (user.isLiveStreaming) {
      liveIndicator = ChannelStatusIndicatorStatus.Live;
      liveIndicatorText = (
        <CoreText data-test-selector={TestSelector.LIVE_INDICATOR_TEXT_SELECTOR} type={TextType.Span}>
          {"live now"}
        </CoreText>
      );
    } else {
      liveIndicatorText = (
        <CoreText type={TextType.Span} data-test-selector={TestSelector.LIVE_INDICATOR_TEXT_SELECTOR}>
          {"not live"}
        </CoreText>
      );
    }

    return (
      <AncestorNameContextProvider ancestorName={AncestorName.UserCard}>
        <Card elevation={3}>
          <StyledLayout className="user-card combo-breaker">
            <UserCardHeader loading={loading} user={user} />
            <StyledLayout color={Color.Alt2} padding={{ top: 0.5, bottom: 1, x: 1 }}>
              {userRoles}
              <StyledLayout
                display={Display.Flex}
                fontSize={FontSize.Size8}
                justifyContent={JustifyContent.Between}
                padding={{ bottom: 2 }}
              >
                <Layout flexOrder={0} fullWidth>
                  <StyledLayout display={Display.Flex} alignItems={AlignItems.Center}>
                    <Layout padding={{ right: 1 }}>
                      <ChannelStatusIndicator
                        status={liveIndicator}
                        data-test-selector={TestSelector.LIVE_INDICATOR_SELECTOR}
                      />
                    </Layout>
                    <Layout padding={{ right: 1 }}>{liveIndicatorText}</Layout>
                  </StyledLayout>
                  <Layout padding={{ right: 1 }}>{accountCreatedText}</Layout>
                </Layout>
                <Layout flexOrder={1}>
                  <IndicatorButton
                    data-track-click="user-card-account-info"
                    data-test-selector={TestSelector.ACCOUNT_INFO_BUTTON_SELECTOR}
                    size={ButtonSize.Small}
                    targetBlank
                    type={ButtonType.Hollow}
                    indicatorIcon={SVGAsset.Popout}
                    linkTo={accountInfoLink(user)}
                  >
                    Account Overview
                  </IndicatorButton>
                </Layout>
              </StyledLayout>
              <UserCardSuspensionDetails loading={loading} user={user} />
            </StyledLayout>
            <StyledLayout
              borderTop
              textAlign={TextAlign.Center}
              color={Color.Alt2}
              display={Display.Flex}
              flexDirection={FlexDirection.Column}
            >
              <StyledLayout borderBottom display={Display.Flex}>
                <IndicatorButton
                  data-track-click="user-card-chatlogs"
                  data-test-selector={TestSelector.CHATLOGS_BUTTON_SELECTOR}
                  disabled={loading}
                  fullWidth
                  indicatorIcon={SVGAsset.Popout}
                  linkTo={chatlogsLink}
                  size={ButtonSize.Small}
                  targetBlank
                  leftAlignText
                  type={ButtonType.Text}
                  padding={{ left: 1 }}
                >
                  Chatlogs
                </IndicatorButton>
                <StyledLayout borderRight />
                <IndicatorButton
                  data-track-click="user-card-channel-page"
                  data-test-selector={TestSelector.CHANNEL_PAGE_BUTTON_SELECTOR}
                  disabled={loading}
                  fullWidth
                  indicatorIcon={SVGAsset.LogoGlitch}
                  linkTo={userProfileLink}
                  size={ButtonSize.Small}
                  targetBlank
                  leftAlignText
                  type={ButtonType.Text}
                  padding={{ left: 1 }}
                >
                  Channel Page
                </IndicatorButton>
              </StyledLayout>
              <StyledLayout borderBottom display={Display.Flex}>
                <IndicatorButton
                  data-track-click="user-card-admin-panel"
                  data-test-selector={TestSelector.ADMIN_PANEL_BUTTON_SELECTOR}
                  disabled={loading}
                  fullWidth
                  indicatorIcon={SVGAsset.Popout}
                  linkTo={adminPanelLink}
                  size={ButtonSize.Small}
                  targetBlank
                  leftAlignText
                  type={ButtonType.Text}
                  padding={{ left: 1 }}
                >
                  Admin Panel
                </IndicatorButton>
                <StyledLayout borderRight />
                <IndicatorButton
                  data-track-click="user-card-vod-manager"
                  data-test-selector={TestSelector.VOD_MANAGER_BUTTON_SELECTOR}
                  disabled={loading}
                  fullWidth
                  indicatorIcon={SVGAsset.LogoGlitch}
                  linkTo={vodManagerLink}
                  size={ButtonSize.Small}
                  targetBlank
                  leftAlignText
                  type={ButtonType.Text}
                  padding={{ left: 1 }}
                >
                  Vod Manager
                </IndicatorButton>
              </StyledLayout>
            </StyledLayout>
          </StyledLayout>
        </Card>
      </AncestorNameContextProvider>
    );
  }
}
