import * as React from "react";

import * as Enzyme from "enzyme";
import { mount, shallow } from "enzyme";
import * as Adapter from "enzyme-adapter-react-16";

import { DetailedUser } from "aegis/models";
import { MockDetailedUser } from "aegis/models/mocks";
import { Avatar, Placeholder } from "twitch-core-ui";
import { ACCEPTABLE_NAME_TEXT_SELECTOR, PublicProps, UserCardHeader } from "./component";
import {
  ACCOUNT_LANG_TEXT_SELECTOR,
  BANNER_IMAGE_SELECTOR,
  CHANNEL_LANG_TEXT_SELECTOR,
  DISPLAY_NAME_TEXT_SELECTOR,
  LOGIN_TEXT_SELECTOR,
  PROFILE_IMAGE_SELECTOR,
  USER_ID_TEXT_SELECTOR
} from "./component";

import "aegis/functionality/date-extensions";

Enzyme.configure({ adapter: new Adapter() });

const setup = (overridesUser?: Partial<DetailedUser>, overridesProps?: Partial<PublicProps>) => {
  const combinedProps: PublicProps = {
    user: MockDetailedUser(overridesUser),
    loading: false,
    ...overridesProps
  };
  return {
    props: combinedProps,
    wrapper: shallow(<UserCardHeader {...combinedProps} />)
  };
};

describe("User Card Header", () => {
  it("Renders loading state and placeholders for the user card header", () => {
    const { wrapper } = setup(undefined, { loading: true });

    expect(wrapper.find({ "data-test-selector": USER_ID_TEXT_SELECTOR }).type()).toEqual(Placeholder);

    expect(wrapper.find({ "data-test-selector": LOGIN_TEXT_SELECTOR })).toHaveLength(0);

    expect(wrapper.find({ "data-test-selector": DISPLAY_NAME_TEXT_SELECTOR }).type()).toEqual(Placeholder);

    expect(wrapper.find({ "data-test-selector": ACCOUNT_LANG_TEXT_SELECTOR }).type()).toEqual(Placeholder);

    expect(wrapper.find({ "data-test-selector": CHANNEL_LANG_TEXT_SELECTOR })).toHaveLength(0);

    expect(wrapper.find({ "data-test-selector": BANNER_IMAGE_SELECTOR })).toHaveLength(0);

    expect(wrapper.find({ "data-test-selector": PROFILE_IMAGE_SELECTOR }).props()).toMatchObject({
      size: 36,
      src: null,
      alt: "Avatar"
    });
  });

  it("Renders a default view of the user card", () => {
    const user = MockDetailedUser({
      channel: {
        language: "en",
        profileBannerImage: "img",
        title: "title",
        mature: false
      }
    });
    const { wrapper } = setup(user);

    expect(mount(wrapper.find({ "data-test-selector": USER_ID_TEXT_SELECTOR }).getElement()).text()).toEqual(user.id);

    expect(mount(wrapper.find({ "data-test-selector": DISPLAY_NAME_TEXT_SELECTOR }).getElement()).text()).toEqual(
      user.displayName
    );

    expect(mount(wrapper.find({ "data-test-selector": LOGIN_TEXT_SELECTOR }).getElement()).text()).toEqual(user.login);
    expect(mount(wrapper.find({ "data-test-selector": ACCEPTABLE_NAME_TEXT_SELECTOR }).getElement()).text()).toEqual(
      "(123.00)"
    );

    expect(mount(wrapper.find({ "data-test-selector": ACCOUNT_LANG_TEXT_SELECTOR }).getElement()).text()).toEqual(
      "en account"
    );

    expect(mount(wrapper.find({ "data-test-selector": CHANNEL_LANG_TEXT_SELECTOR }).getElement()).text()).toEqual(
      " ● en channel"
    );

    expect(wrapper.find({ "data-test-selector": BANNER_IMAGE_SELECTOR }).type()).toEqual("img");

    expect(wrapper.find({ "data-test-selector": PROFILE_IMAGE_SELECTOR }).type()).toEqual(Avatar);
    expect(wrapper.find({ "data-test-selector": PROFILE_IMAGE_SELECTOR }).props()).toMatchObject({
      alt: "Avatar",
      size: 36,
      src: user.profileImage
    });
  });

  it("Renders the user card when no channel information is retrieved", () => {
    const { wrapper } = setup({ channel: undefined });
    expect(wrapper.find({ "data-test-selector": CHANNEL_LANG_TEXT_SELECTOR })).toHaveLength(0);

    expect(wrapper.find({ "data-test-selector": BANNER_IMAGE_SELECTOR })).toHaveLength(0);
  });

  it("Renders the user card when display name and login are the same", () => {
    const { wrapper } = setup({ displayName: "DIFFERENT_CASE_NAME", login: "different_case_name" });

    expect(mount(wrapper.find({ "data-test-selector": DISPLAY_NAME_TEXT_SELECTOR }).getElement()).text()).toEqual(
      "DIFFERENT_CASE_NAME"
    );

    expect(wrapper.find({ "data-test-selector": LOGIN_TEXT_SELECTOR })).toHaveLength(0);
  });
});
