import * as React from "react";

import { WithTranslatePopup } from "aegis/features/translate";
import { TranslateIconPlacement } from "aegis/features/translate/component";
import { DetailedUser } from "aegis/models";

import {
  Avatar,
  Background,
  BorderRadius,
  Color,
  CoreText,
  Display,
  FlexDirection,
  FontSize,
  JustifyContent,
  Layout,
  Placeholder,
  Position,
  StyledLayout,
  TextType,
  Tooltip
} from "twitch-core-ui";

import "./styles.scss";

export const USER_ID_TEXT_SELECTOR = "user-id-text-selector";
export const LOGIN_TEXT_SELECTOR = "login-text-selector";
export const DISPLAY_NAME_TEXT_SELECTOR = "display-name-text-selector";
export const ACCOUNT_LANG_TEXT_SELECTOR = "account-lang-text-selector";
export const CHANNEL_LANG_TEXT_SELECTOR = "channel-lang-text-selector";
export const BANNER_IMAGE_SELECTOR = "banner-image-selector";
export const PROFILE_IMAGE_SELECTOR = "profile-image-selector";
export const ACCEPTABLE_NAME_TEXT_SELECTOR = "acceptable-name-text-selector";

export interface PublicProps {
  user: DetailedUser;
  loading?: boolean;
}

export const UserCardHeader: React.SFC<PublicProps> = props => {
  let banner = null;
  let profileImage = null;
  let channelLangText = null;
  let accountLangText = <Placeholder data-test-selector={ACCOUNT_LANG_TEXT_SELECTOR} />;
  let userIDText = <Placeholder data-test-selector={USER_ID_TEXT_SELECTOR} />;
  let displayNameText = <Placeholder data-test-selector={DISPLAY_NAME_TEXT_SELECTOR} />;
  let offensiveLevelText = <Placeholder data-test-selector={ACCEPTABLE_NAME_TEXT_SELECTOR} />;
  let loginText = null;

  const { loading, user } = props;
  if (!loading) {
    if (user.channel) {
      if (user.channel.profileBannerImage) {
        banner = (
          <img
            alt="Banner"
            className="banner"
            src={user.channel.profileBannerImage}
            data-test-selector={BANNER_IMAGE_SELECTOR}
          />
        );
      }
      if (user.channel.language) {
        channelLangText = (
          <Layout data-test-selector={CHANNEL_LANG_TEXT_SELECTOR} display={Display.InlineBlock}>
            <CoreText key="separator" type={TextType.Span}>
              {" ● "}
            </CoreText>
            <span key="accountLang" className="semi-bold tw-upcase">
              {user.channel.language}
            </span>
            <span key="label" className="semi-thin">
              {" channel"}
            </span>
          </Layout>
        );
      }
    }
    profileImage = user.profileImage;
    accountLangText = (
      <Layout data-test-selector={ACCOUNT_LANG_TEXT_SELECTOR} display={Display.InlineBlock} margin={{ right: 0.5 }}>
        <span key="accountLang" className="semi-bold tw-upcase">
          {user.language}
        </span>
        <span key="label" className="semi-thin">
          {" account"}
        </span>
      </Layout>
    );

    let displayNameLabel: string = "display name";
    if (user.displayName.toLowerCase() !== user.login.toLowerCase()) {
      loginText = (
        <Layout margin={{ left: 0.5 }} display={Display.Inline}>
          {" ("}
          <CoreText type={TextType.Span} data-test-selector={LOGIN_TEXT_SELECTOR}>
            {user.login}
          </CoreText>
          <WithTranslatePopup text={user.login} iconPlacement={TranslateIconPlacement.After} label="username">
            <CoreText />
          </WithTranslatePopup>
          {")"}
        </Layout>
      );
    } else {
      displayNameLabel = "username";
    }
    displayNameText = (
      <>
        <CoreText type={TextType.Span} data-test-selector={DISPLAY_NAME_TEXT_SELECTOR}>
          {user.displayName}
        </CoreText>
        <WithTranslatePopup
          text={user.displayName}
          iconPlacement={TranslateIconPlacement.After}
          label={displayNameLabel}
        >
          <CoreText />
        </WithTranslatePopup>
      </>
    );

    userIDText = (
      <CoreText type={TextType.Span} data-test-selector={USER_ID_TEXT_SELECTOR}>
        {user.id}
      </CoreText>
    );
    offensiveLevelText = (
      <Tooltip label="Username offensive level">
        <CoreText
          type={TextType.Span}
          data-test-selector={ACCEPTABLE_NAME_TEXT_SELECTOR}
          color={user.acceptableNameConfidence! < 0.8 ? Color.Overlay : Color.Error}
          fontSize={FontSize.Size8}
        >
          {`(${user.acceptableNameConfidence!.toFixed(2)})`}
        </CoreText>
      </Tooltip>
    );
  }

  return (
    <StyledLayout className="user-card__header">
      {banner}
      <StyledLayout
        className="user-card__overlay"
        attachTop
        background={Background.Overlay}
        fullWidth
        position={Position.Absolute}
      >
        <StyledLayout margin={{ x: 1, top: 4, bottom: 0.5 }} color={Color.Overlay}>
          <Layout display={Display.Flex}>
            <div className="avatar-container">
              <Avatar
                size={36}
                src={profileImage}
                alt="Avatar"
                borderRadius={BorderRadius.None}
                data-test-selector={PROFILE_IMAGE_SELECTOR}
              />
            </div>
            <StyledLayout fullWidth display={Display.Flex} flexDirection={FlexDirection.Column}>
              <CoreText type={TextType.Span}>
                <span className={".semi-bold"}>{displayNameText}</span>
                <span className={".semi-thin"}>
                  {loginText} {offensiveLevelText}
                </span>
              </CoreText>
              <CoreText fontSize={FontSize.Size8}>{userIDText}</CoreText>
            </StyledLayout>
          </Layout>
          <StyledLayout
            className="lang-container"
            display={Display.Flex}
            fontSize={FontSize.Size8}
            justifyContent={JustifyContent.Between}
          >
            <Layout flexOrder={0}>
              {accountLangText}
              {channelLangText}
            </Layout>
          </StyledLayout>
        </StyledLayout>
      </StyledLayout>
    </StyledLayout>
  );
};
