import * as React from "react";

import * as Enzyme from "enzyme";
import { mount, shallow } from "enzyme";
import * as Adapter from "enzyme-adapter-react-16";

import { DetailedUser } from "aegis/models";
import { MockDetailedUser } from "aegis/models/mocks";
import { Placeholder } from "twitch-core-ui";
import { PublicProps, UserCardSuspensionDetails } from "./component";
import {
  BANNED_UNTIL_TEXT_SELECTOR,
  DMCA_COUNT_TEXT_SELECTOR,
  IS_SUSPENDED_TEXT_SELECTOR,
  NUM_REPORTED_LAST_DAY_TEXT_SELECTOR,
  NUM_REPORTS_SENT_LAST_DAY_TEXT_SELECTOR,
  NUM_WARNED_LAST_DAY_TEXT_SELECTOR,
  SUSPENSION_LINK_SELECTOR,
  TOS_COUNT_TEXT_SELECTOR
} from "./component";

import "aegis/functionality/date-extensions";

Enzyme.configure({ adapter: new Adapter() });

const setup = (overridesUser?: Partial<DetailedUser>, overridesProps?: Partial<PublicProps>) => {
  const combinedProps: PublicProps = {
    user: MockDetailedUser(overridesUser),
    loading: false,
    ...overridesProps
  };
  return {
    props: combinedProps,
    wrapper: shallow(<UserCardSuspensionDetails {...combinedProps} />)
  };
};

describe("User Card Header", () => {
  it("Renders loading state and placeholders for the user card suspension details", () => {
    const { wrapper } = setup(undefined, { loading: true });

    expect(wrapper.find({ "data-test-selector": IS_SUSPENDED_TEXT_SELECTOR }).type()).toEqual(Placeholder);

    expect(wrapper.find({ "data-test-selector": TOS_COUNT_TEXT_SELECTOR }).type()).toEqual(Placeholder);

    expect(wrapper.find({ "data-test-selector": DMCA_COUNT_TEXT_SELECTOR }).type()).toEqual(Placeholder);

    expect(wrapper.find({ "data-test-selector": NUM_REPORTED_LAST_DAY_TEXT_SELECTOR }).type()).toEqual(Placeholder);

    expect(wrapper.find({ "data-test-selector": NUM_REPORTS_SENT_LAST_DAY_TEXT_SELECTOR }).type()).toEqual(Placeholder);

    expect(wrapper.find({ "data-test-selector": NUM_WARNED_LAST_DAY_TEXT_SELECTOR }).type()).toEqual(Placeholder);

    expect(wrapper.find({ "data-test-selector": BANNED_UNTIL_TEXT_SELECTOR })).toHaveLength(0);

    expect(wrapper.find({ "data-test-selector": SUSPENSION_LINK_SELECTOR })).toHaveLength(0);
  });

  it("Renders a default view of the user card suspension details", () => {
    const user = MockDetailedUser();
    const { wrapper } = setup(user);

    expect(mount(wrapper.find({ "data-test-selector": IS_SUSPENDED_TEXT_SELECTOR }).getElement()).text()).toEqual(
      "Yes"
    );

    expect(mount(wrapper.find({ "data-test-selector": TOS_COUNT_TEXT_SELECTOR }).getElement()).text()).toEqual("3");

    expect(mount(wrapper.find({ "data-test-selector": DMCA_COUNT_TEXT_SELECTOR }).getElement()).text()).toEqual("1");

    expect(
      mount(wrapper.find({ "data-test-selector": NUM_REPORTED_LAST_DAY_TEXT_SELECTOR }).getElement()).text()
    ).toEqual(`${user.reportedCount}`);

    expect(
      mount(wrapper.find({ "data-test-selector": NUM_REPORTS_SENT_LAST_DAY_TEXT_SELECTOR }).getElement()).text()
    ).toEqual(`${user.reportSentCount}`);

    expect(
      mount(wrapper.find({ "data-test-selector": NUM_WARNED_LAST_DAY_TEXT_SELECTOR }).getElement()).text()
    ).toEqual(`${user.warnedCount}`);

    expect(wrapper.find({ "data-test-selector": BANNED_UNTIL_TEXT_SELECTOR })).toHaveLength(0);

    expect(wrapper.find({ "data-test-selector": SUSPENSION_LINK_SELECTOR })).toHaveLength(1);
  });

  it("Renders the user card suspension details when suspension date and id are given", () => {
    const { wrapper } = setup({ bannedUntil: "2020-01-01T01:13:11+00:00" });

    expect(mount(wrapper.find({ "data-test-selector": BANNED_UNTIL_TEXT_SELECTOR }).getElement()).text()).toEqual(
      "Until 2020-01-01 01:13:11Z"
    );

    expect(wrapper.find({ "data-test-selector": SUSPENSION_LINK_SELECTOR }).props().to).toBe(
      "https://leviathan.internal.twitch.tv/suspensions/search?suspensions%5Btarget_user_id%5D=id"
    );
    expect(wrapper.find({ "data-test-selector": SUSPENSION_LINK_SELECTOR }).props().targetBlank).toBe(true);
  });
});
