import * as React from "react";

import { DetailedUser } from "aegis/models";
import { Color, Column, CoreLink, CoreText, FontSize, Grid, Layout, Placeholder, TextType } from "twitch-core-ui";

export const IS_ESCALATE_PARTNER_SELECTOR = "is-escalate-partner-text-selector";
export const IS_SUSPENDED_TEXT_SELECTOR = "is-suspended-text-selector";
export const TOS_COUNT_TEXT_SELECTOR = "tos-count-text-selector";
export const DMCA_COUNT_TEXT_SELECTOR = "dmca-count-text-selector";
export const NUM_REPORTED_LAST_DAY_TEXT_SELECTOR = "num-reported-last-day-text-selector";
export const NUM_WARNED_LAST_DAY_TEXT_SELECTOR = "num-warned-last-day-text-selector";
export const NUM_REPORTS_SENT_LAST_DAY_TEXT_SELECTOR = "num-reports-sent-last-day-text-selector";
export const BANNED_UNTIL_TEXT_SELECTOR = "banned-until-text-selector";
export const SUSPENSION_LINK_SELECTOR = "suspension-link-selector";

export interface PublicProps {
  user: DetailedUser;
  loading?: boolean;
}

export const UserCardSuspensionDetails: React.SFC<PublicProps> = props => {
  let isSuspended: JSX.Element | string = <Placeholder data-test-selector={IS_SUSPENDED_TEXT_SELECTOR} />;
  let tosCount: JSX.Element | string = <Placeholder data-test-selector={TOS_COUNT_TEXT_SELECTOR} />;
  let dmcaCount: JSX.Element | string = <Placeholder data-test-selector={DMCA_COUNT_TEXT_SELECTOR} />;
  let reportedCount: JSX.Element | string = <Placeholder data-test-selector={NUM_REPORTED_LAST_DAY_TEXT_SELECTOR} />;
  let warnedCount: JSX.Element | string = <Placeholder data-test-selector={NUM_WARNED_LAST_DAY_TEXT_SELECTOR} />;
  let reportSentCount: JSX.Element | string = (
    <Placeholder data-test-selector={NUM_REPORTS_SENT_LAST_DAY_TEXT_SELECTOR} />
  );

  let bannedUntil: JSX.Element | null = null;
  let suspensionDetail: JSX.Element | null = null;

  const { loading, user } = props;

  if (!loading) {
    isSuspended = (
      <CoreText type={TextType.Span} data-test-selector={IS_SUSPENDED_TEXT_SELECTOR}>
        {user.isBanned || user.isDMCABanned ? "Yes" : "No"}
      </CoreText>
    );
    tosCount = (
      <CoreText type={TextType.Span} data-test-selector={TOS_COUNT_TEXT_SELECTOR}>
        {user.tosViolationCount}
      </CoreText>
    );
    dmcaCount = (
      <CoreText type={TextType.Span} data-test-selector={DMCA_COUNT_TEXT_SELECTOR}>
        {user.dmcaViolationCount}
      </CoreText>
    );
    reportedCount = (
      <CoreText type={TextType.Span} data-test-selector={NUM_REPORTED_LAST_DAY_TEXT_SELECTOR}>
        {user.reportedCount}
      </CoreText>
    );
    warnedCount = (
      <CoreText type={TextType.Span} data-test-selector={NUM_WARNED_LAST_DAY_TEXT_SELECTOR}>
        {user.warnedCount}
      </CoreText>
    );
    reportSentCount = (
      <CoreText type={TextType.Span} data-test-selector={NUM_REPORTS_SENT_LAST_DAY_TEXT_SELECTOR}>
        {user.reportSentCount}
      </CoreText>
    );

    if (user.bannedUntil) {
      const timestamp = Date.parse(user.bannedUntil);
      bannedUntil = (
        <CoreText fontSize={FontSize.Size8} data-test-selector={BANNED_UNTIL_TEXT_SELECTOR}>
          Until {new Date(timestamp).toDisplayFormat()}
        </CoreText>
      );
    }

    if (user.isBanned || user.isDMCABanned) {
      suspensionDetail = (
        <Layout>
          <CoreLink
            to={"https://leviathan.internal.twitch.tv/suspensions/search?suspensions%5Btarget_user_id%5D=" + user.id}
            targetBlank
            data-test-selector={SUSPENSION_LINK_SELECTOR}
          >
            {"View Suspensions"}
          </CoreLink>
        </Layout>
      );
    }
  }

  return (
    <Layout>
      <Layout margin={{ bottom: 1 }}>
        <Grid>
          <Column cols={6}>Currently Suspended</Column>
          <Column cols={6}>
            <Layout>
              <CoreText bold color={Color.Base}>
                {isSuspended}
              </CoreText>
              {bannedUntil}
              {suspensionDetail}
            </Layout>
          </Column>
        </Grid>
      </Layout>
      <Layout margin={{ bottom: 1 }}>
        <Grid>
          <Column cols={6}>TOS Suspensions</Column>
          <Column cols={6}>{tosCount}</Column>
        </Grid>
      </Layout>
      <Layout margin={{ bottom: 1 }}>
        <Grid>
          <Column cols={6}>DMCA Suspensions</Column>
          <Column cols={6}>{dmcaCount}</Column>
        </Grid>
      </Layout>
      <Layout margin={{ bottom: 1 }}>
        <Grid>
          <Column cols={6}>Warnings</Column>
          <Column cols={6}>{warnedCount}</Column>
        </Grid>
      </Layout>
      <Layout margin={{ bottom: 1 }}>
        <Grid>
          <Column cols={6}>Past 24h times reported</Column>
          <Column cols={6}>
            <CoreText>{reportedCount}</CoreText>
          </Column>
        </Grid>
      </Layout>

      <Layout margin={{ bottom: 1 }}>
        <Grid>
          <Column cols={6}>Past 24h reports sent</Column>
          <Column cols={6}>
            <CoreText>{reportSentCount}</CoreText>
          </Column>
        </Grid>
      </Layout>
    </Layout>
  );
};
