import * as React from "react";

import { UserRole, UserRoleTagMapping } from "aegis/models";
import { AlignItems, Display, FlexWrap, Icon, Layout, Pill, PillType, SVGAsset } from "twitch-core-ui";

import "./styles.scss";

export interface Props {
  roles?: UserRole[];
  expanded?: boolean;
  pill?: boolean;
}

export class UserRoleTags extends React.PureComponent<Props> {
  public render() {
    if (!this.props.roles || this.props.roles.length === 0) {
      return null;
    }
    const { expanded, roles, pill } = this.props;
    let elements: JSX.Element[] = [];
    if (expanded) {
      elements = this.props.roles.map((r: UserRole, index: number) => {
        return (
          <Layout key={index} padding={{ right: 1, bottom: 0.5 }} display={Display.InlineFlex}>
            {this.getUserRole(r, pill)}
          </Layout>
        );
      });
    } else {
      if (roles.length > 0) {
        elements.push(this.getUserRole(roles[0], pill));
      }
      if (roles.length > 1) {
        const moreRoles = `+${roles.length - 1}`;
        if (pill) {
          elements.push(this.userRolePill(`${moreRoles}`));
        } else {
          elements.push(this.userRoleTag({ text: `${moreRoles}` }));
        }
      }
    }

    return (
      <Layout display={Display.Flex} flexWrap={FlexWrap.Wrap} alignItems={AlignItems.Center}>
        {elements}
      </Layout>
    );
  }

  private getUserRole(userRole: UserRole, pill?: boolean): JSX.Element {
    const roleValue = UserRoleTagMapping.get(userRole);
    if (roleValue) {
      if (pill) {
        return this.userRolePill(roleValue.text);
      } else {
        return <UserRoleTag {...roleValue} key={userRole} />;
      }
    } else {
      console.warn("Unrecognized user role: ", userRole);
      return this.userRoleTag({ text: userRole, icon: SVGAsset.QuestionMark });
    }
  }

  private userRoleTag = (role: { text: string; icon?: SVGAsset }) => {
    return <UserRoleTag {...role} key={role.text} />;
  };

  private userRolePill = (text: string) => {
    return (
      <Layout className="user-role-tag__pill" key={text}>
        <Pill type={PillType.Overlay} label={text} />
      </Layout>
    );
  };
}

interface TagProps {
  text: string;
  icon?: SVGAsset;
}

export const UserRoleTag: React.SFC<TagProps> = ({ text, icon }) => {
  return (
    <Layout display={Display.Flex} className="user-role-tag__tag" key={text} alignItems={AlignItems.Center}>
      {icon && <Icon asset={icon} />}
      <Layout padding={{ left: 0.5 }}>{text}</Layout>
    </Layout>
  );
};
