import * as React from "react";

import { UserCard } from "aegis/features/user-card";
import { UserRoleTags } from "aegis/features/user-role-tags";
import { ToggleBalloonWrapper } from "aegis/functionality/components/toggle-balloon-wrapper";
import { Badge, getNameAsDisplayed, User } from "aegis/models";
import {
  AlignItems,
  Background,
  Balloon,
  BalloonDirection,
  Color,
  CoreImage,
  CoreText,
  Display,
  FlexWrap,
  Interactable,
  Layout,
  Pill,
  PillType,
  StyledLayout,
  TextType,
  Tooltip,
  VerticalAlign
} from "twitch-core-ui";
import { ExpandedUserView } from "./components/expanded-view";

import "./style.scss";

export interface Props {
  // Passed in from container
  user?: User;
}

export interface PassthroughProps {
  // overrides shown badges
  badges?: Badge[];
  showStatus?: boolean;
  showType?: boolean;
  inline?: boolean;
  className?: string;
  balloonDirection?: BalloonDirection;
}

export const UserDisplaySelector = "user.display";
export const UserIDSelector = "user.id";

export class UserComponent extends React.Component<Props & PassthroughProps> {
  public render() {
    const { badges, user, balloonDirection } = this.props;
    if (!user) {
      return (
        <CoreText type={TextType.Span} color={Color.Error}>
          User not found
        </CoreText>
      );
    }

    if (!this.props.inline) {
      return (
        <ExpandedUserView user={user} className={this.props.className} balloonDirection={balloonDirection}>
          {this.props.children}
        </ExpandedUserView>
      );
    }

    const badgeElements: React.ReactNode[] = [];
    if (Array.isArray(badges)) {
      badges.forEach((badge, index) => {
        const margin = index === badges.length - 1 ? {} : { right: 0.5 };
        // Skip badges that we don't have information for
        if (badge.imageURL && badge.title) {
          badgeElements.push(
            <Layout
              key={badge.imageURL}
              display={Display.Inline}
              verticalAlign={VerticalAlign.Middle}
              flexGrow={0}
              flexShrink={0}
              {...margin}
            >
              <Tooltip display={Display.Inline} key={badge.title} label={badge.title}>
                <CoreImage alt={badge.title} src={badge.imageURL} />
              </Tooltip>
            </Layout>
          );
        }
      });
    } else if (typeof badges === "undefined") {
      if (this.props.showType && user.roles) {
        badgeElements.push(<UserRoleTags key="roles" roles={user.roles} pill />);
      }
      if (this.props.showStatus) {
        badgeElements.push(this.userStatusPill(user));
      }
    }

    return (
      <StyledLayout display={Display.Flex} flexWrap={FlexWrap.Wrap}>
        {badgeElements}
        <Layout key="user-name" padding={{ x: 0.5 }} data-test-selector={UserDisplaySelector}>
          <CoreText type={TextType.Span}>{getNameAsDisplayed(user)}</CoreText>
        </Layout>
        <Layout key="user-id" padding={{ x: 0.5 }} data-test-selector={UserIDSelector}>
          <CoreText type={TextType.Span}>({user.id})</CoreText>
        </Layout>
        <ToggleBalloonWrapper key="user-card-balloon">
          <Interactable data-track-click="user-inline-show-card">
            <Layout padding={{ x: 0.5 }}>
              <i className="fa fa-caret-down" data-track-click="user-inline-show-card" />
            </Layout>
          </Interactable>
          <Balloon
            show
            direction={balloonDirection ? balloonDirection : BalloonDirection.BottomCenter}
            tailBackground={Background.Overlay}
          >
            <UserCard userID={user.id} />
          </Balloon>
        </ToggleBalloonWrapper>
      </StyledLayout>
    );
  }

  private userStatusPill(user: User) {
    // Invalid user, show nothing
    if (!user) {
      return null;
    }

    const userPills: React.ReactNode[] = [];

    if (user.isBanned) {
      userPills.push(
        <Layout key="banned" className="user-status-tag__pill">
          <Pill type={PillType.Alert} label="Banned" />
        </Layout>
      );
    }
    if (user.isDMCABanned) {
      userPills.push(
        <Layout key="dmca" className="user-status-tag__pill">
          <Pill type={PillType.Alert} label="DMCA" />
        </Layout>
      );
    }
    if (user.isDeleted) {
      userPills.push(
        <Layout key="deleted" className="user-status-tag__pill">
          <Pill type={PillType.Alert} label="Deleted" />
        </Layout>
      );
    }
    if (userPills.length === 0) {
      userPills.push(
        <Layout key="active" className="user-status-tag__pill">
          <Pill type={PillType.Success} label="Active" />
        </Layout>
      );
    }

    if (userPills.length === 0) {
      return null;
    }

    return (
      <Layout key="user-status-tags" display={Display.Flex} flexWrap={FlexWrap.NoWrap} alignItems={AlignItems.Center}>
        {userPills}
      </Layout>
    );
  }
}
