import * as React from "react";

import gql from "graphql-tag";

import { Query } from "aegis/functionality/apollo-wrapper";
import { CoreText } from "twitch-core-ui";
import { OnEscalateList } from "./component";

export type Response = {
  user: {
    isEscalatePartner: boolean;
  };
};

export const FETCH_USER_IS_ESCALATE_PARTNER = gql`
  query user_escalate_partner($userID: ID) {
    user(id: $userID) {
      id
      isEscalatePartner
    }
  }
`;

interface QueryInput {
  userID: string;
}

export interface DataProps {
  onCompleted?: (isEscalatePartner: boolean) => void;
}

export type EscalateStatusProps = QueryInput & DataProps;

export class EscalatePartnerQuery extends Query<Response, QueryInput> {}

export const EscalateStatus: React.SFC<EscalateStatusProps> = ({ userID, onCompleted }) => {
  return (
    <EscalatePartnerQuery
      displayName="EscalateStatus"
      query={FETCH_USER_IS_ESCALATE_PARTNER}
      variables={{ userID }}
      notifyOnNetworkStatusChange
      onCompleted={data => {
        if (onCompleted) {
          onCompleted(data && data.user.isEscalatePartner);
        }
      }}
    >
      {({ loading, data, error }) => {
        if (loading) {
          return <div />;
        }

        if (error) {
          return <CoreText>Error Loading EnforcementList: {error ? error.message : ""}</CoreText>;
        }

        if (!!data) {
          return <OnEscalateList IsEscalatePartner={data.user.isEscalatePartner} />;
        }
      }}
    </EscalatePartnerQuery>
  );
};
