import * as React from "react";

import gql from "graphql-tag";

import * as Fragments from "aegis/fragments";
import { Query } from "aegis/functionality/apollo-wrapper";
import * as Models from "aegis/models";
import { Color, CoreText, LoadingSpinner } from "twitch-core-ui";
import { PassthroughProps, UserComponent } from "./component";

export type Response = {
  user: Models.User;
};

export const UserQuery = gql`
  query user($id: ID, $login: String) {
    user(id: $id, login: $login) {
      ...UserFields
    }
  }
  ${Fragments.UserFields}
`;

type InputProps = {
  login?: string;
  id?: string;
};

type SharedProps = {
  onChange?: (user: Models.User) => void;
  cacheOnly?: boolean;
} & PassthroughProps;

export type WithUserID = { userID: string } & SharedProps;
export type WithLogin = { login: string } & SharedProps;
export type WithUserModel = { user: Models.User } & SharedProps;

// This requires the container receives EITHER userID OR login
type ContainerProps = WithUserID | WithLogin | WithUserModel;

const hasUserID = (props: ContainerProps): props is WithUserID => !!(props as WithUserID).userID;
const hasLogin = (props: ContainerProps): props is WithLogin => !!(props as WithLogin).login;
const hasUserModel = (props: ContainerProps): props is WithUserModel => !!(props as WithUserModel).user;

export class UserQueryWrapper extends Query<Response, InputProps> {}

export class User extends React.Component<ContainerProps> {
  public render() {
    const { cacheOnly } = this.props;
    const variables: InputProps = {};
    if (hasUserID(this.props)) {
      variables.id = this.props.userID;
    }
    if (hasLogin(this.props)) {
      variables.login = this.props.login;
    }
    if (hasUserModel(this.props)) {
      return <UserComponent {...this.props}>{this.props.children}</UserComponent>;
    }

    return (
      <UserQueryWrapper
        displayName="User"
        query={UserQuery}
        variables={variables}
        fetchPolicy={cacheOnly ? "cache-only" : "cache-first"}
        notifyOnNetworkStatusChange // https://github.com/apollographql/react-apollo/issues/2177#issuecomment-428666098
        onCompleted={this.onComplete}
      >
        {({ loading, error, data }) => {
          if (loading) {
            return <LoadingSpinner />;
          }
          if (error) {
            console.error(error);
            return <CoreText color={Color.Error}>Error: {error.message}</CoreText>;
          }
          if (!data) {
            console.error("User Not Found");
            return <div>User Not Found</div>;
          }

          return (
            <UserComponent {...this.props} user={data.user}>
              {this.props.children}
            </UserComponent>
          );
        }}
      </UserQueryWrapper>
    );
  }

  private onComplete = (data: Response) => {
    if (data.user && this.props.onChange) {
      this.props.onChange(data.user);
    }
  };
}
