import * as React from "react";

import ApolloClient, { ApolloQueryResult } from "apollo-client";
import gql from "graphql-tag";
import { action, observable } from "mobx";
import { observer } from "mobx-react";
import { compose, withApollo } from "react-apollo";

import { apolloErrorHandler, errorNotification } from "aegis/functionality/error-notifications";
import { VodPopoutButtonComponent } from "./component";

export const VodPopoutButtonQuery = gql`
  query VodAtTimestamp($channelID: ID!, $timestamp: Time!) {
    vod: vodAtTimestamp(channelID: $channelID, timestamp: $timestamp) {
      twitchVideoPageURL
    }
  }
`;

type Props = {
  channelID: string;
  timestamp: number;
  tooltip: string;
  client: ApolloClient<{}>;
};

type VodLinkFetchResult = {
  vod: {
    twitchVideoPageURL: string;
  };
};

@observer
export class VodPopoutButtonContainer extends React.Component<Props> {
  @observable private errorTooltip?: string;
  @observable private disableButton: boolean = false;

  public render() {
    return (
      <VodPopoutButtonComponent
        tooltip={this.errorTooltip ? this.errorTooltip : this.props.tooltip}
        onClick={this.onClick}
        disabled={this.disableButton}
      />
    );
  }

  @action
  private onClick = () => {
    this.props.client
      .query({
        query: VodPopoutButtonQuery,
        variables: { channelID: this.props.channelID, timestamp: this.props.timestamp }
      })
      .then(({ data }: ApolloQueryResult<VodLinkFetchResult>) => {
        if (data && data.vod && data.vod.twitchVideoPageURL) {
          window.open(data.vod.twitchVideoPageURL);
        } else {
          errorNotification("Can't find a vod for that link");
          this.errorTooltip = "Vod does not exist";
          this.disableButton = true;
        }
      })
      .catch(error => {
        apolloErrorHandler("Error when looking up that VOD")(error);
        this.errorTooltip = "An error occurred while looking up the vod. Please retry";
      });
  };
}

export const VodPopoutButton = compose(withApollo)(VodPopoutButtonContainer);
