import * as React from "react";

import { action, computed, observable } from "mobx";
import { observer } from "mobx-react";

import { Vod } from "aegis/models";
import { DeleteVodAndClipsButton } from "./delete-vod-and-clips-button";

import {
  Button,
  ButtonType,
  Column,
  CoreLink,
  CoreText,
  Display,
  FlexDirection,
  FormGroup,
  Grid,
  JustifyContent,
  Layout,
  Overflow,
  ResponsiveWrapper,
  SegmentedButton,
  SegmentedButtonOption,
  Tooltip,
  TooltipDirection
} from "twitch-core-ui";

import "./styles.scss";

interface Props {
  vod: Vod;
}

const VodEmbedButtonGroupPrefix = "vod-video-embed--";

export enum DisplayOption {
  Twitch = "twitch",
  Unmuted = "unmuted"
}

export enum TestSelectors {
  TwitchEmbedButton = "vod-video-embed--twitch-embed-button",
  UnmutedEmbedButton = "vod-video-embed--unmuted-embed-button",
  TwitchEmbedIFrame = "vod-video-embed--twitch-embed-iframe",
  UnmutedEmbedIFrame = "vod-video-embed--unmuted-embed-iframe",
  ClipsOverviewURL = "vod-video-embed--clips-overview-url"
}

@observer
export class VodVideoEmbed extends React.Component<Props> {
  @observable private videoOption = DisplayOption.Twitch;
  @observable private showMoreDesc = false;

  public componentDidMount() {
    if (this.props.vod.isDeleted) {
      this.videoOption = DisplayOption.Unmuted;
    }
  }

  public render() {
    const { vod } = this.props;

    const buttonGroupName = VodEmbedButtonGroupPrefix + vod.id;
    return (
      <Layout
        fullWidth
        display={Display.Flex}
        justifyContent={JustifyContent.Center}
        flexDirection={FlexDirection.Column}
      >
        <ResponsiveWrapper>
          <Grid justifyContent={JustifyContent.Start}>
            <Column cols={{ default: 6, xs: 6, sm: 6, md: 4, lg: 4, xl: 2, xxl: 2 }}>
              <Layout margin={{ y: 1 }}>
                <FormGroup label="VOD ID">{vod.id}</FormGroup>
              </Layout>
            </Column>
            <Column cols={{ default: 6, xs: 6, sm: 6, md: 4, lg: 4, xl: 2, xxl: 2 }}>
              <Layout margin={{ y: 1 }}>
                <FormGroup label="Collection">{vod.collection ? vod.collection : "None"}</FormGroup>
              </Layout>
            </Column>
            <Column cols={{ default: 6, xs: 6, sm: 6, md: 4, lg: 4, xl: 2, xxl: 2 }}>
              <Layout margin={{ y: 1 }}>
                <FormGroup label="Created At">
                  {vod.createdAt ? new Date(vod.createdAt).toDisplayFormat() : "Unknown"}
                </FormGroup>
              </Layout>
            </Column>
            <Column cols={{ default: 6, xs: 6, sm: 6, md: 4, lg: 4, xl: 2, xxl: 2 }}>
              <Layout margin={{ y: 1 }}>
                <FormGroup label="Deleted">{vod.isDeleted ? "Yes" : "No"}</FormGroup>
              </Layout>
            </Column>
            <Column cols={{ default: 6, xs: 6, sm: 6, md: 4, lg: 4, xl: 2, xxl: 2 }}>
              <Layout margin={{ y: 1 }}>
                <FormGroup label="Offset">
                  <Tooltip label="User reported at this timestamp" direction={TooltipDirection.Bottom}>
                    {vod.offset ? vod.offset : "None"}
                  </Tooltip>
                </FormGroup>
              </Layout>
            </Column>
            <Column cols={{ default: 6, xs: 6, sm: 6, md: 4, lg: 4, xl: 2, xxl: 2 }}>
              <Layout margin={{ y: 1 }}>
                <FormGroup label="Title">{vod.title}</FormGroup>
              </Layout>
            </Column>
            {vod.description && (
              <Column cols={this.showMoreDesc ? 12 : { default: 6, xs: 6, sm: 6, md: 4, lg: 4, xl: 2, xxl: 2 }}>
                <Layout margin={{ y: 1 }}>
                  <FormGroup label="Description" />
                  <Layout overflow={Overflow.Auto} className="vod-video-embed__description">
                    <CoreText ellipsis={!this.showMoreDesc}>{vod.description}</CoreText>
                  </Layout>
                  <CoreLink onClick={() => (this.showMoreDesc = !this.showMoreDesc)}>
                    {this.showMoreDesc ? "Show less" : "Show more"}
                  </CoreLink>
                </Layout>
              </Column>
            )}
          </Grid>
        </ResponsiveWrapper>
        <Layout
          display={Display.Flex}
          justifyContent={JustifyContent.Center}
          flexDirection={FlexDirection.Column}
          className="vod-video-embed__video"
        >
          <Layout
            display={this.shouldShowTwitchEmbed ? Display.Block : Display.Hide}
            data-test-selector={TestSelectors.TwitchEmbedIFrame}
          >
            <iframe width={480} height={270} src={vod.url} allowFullScreen={true} />
          </Layout>
          <Layout
            display={this.shouldShowUnmutedEmbed ? Display.Block : Display.Hide}
            data-test-selector={TestSelectors.UnmutedEmbedIFrame}
          >
            <iframe width={480} height={270} src={vod.unmutedURL} allowFullScreen={true} />
          </Layout>

          <Layout display={Display.Flex} justifyContent={JustifyContent.Between}>
            <Layout>
              <SegmentedButton>
                <SegmentedButtonOption
                  data-track-click="vod-twitch"
                  name={buttonGroupName}
                  label="Twitch"
                  checked={this.shouldShowTwitchEmbed}
                  value={DisplayOption.Twitch}
                  onChange={this.updateEmbed}
                  data-test-selector={TestSelectors.TwitchEmbedButton}
                />
                <SegmentedButtonOption
                  data-track-click="vod-unmuted"
                  name={buttonGroupName}
                  label="Unmuted"
                  checked={this.shouldShowUnmutedEmbed}
                  value={DisplayOption.Unmuted}
                  onChange={this.updateEmbed}
                  data-test-selector={TestSelectors.UnmutedEmbedButton}
                />
              </SegmentedButton>
            </Layout>

            <Tooltip label="Clips around vod creation time + offset" direction={TooltipDirection.Left}>
              <Button
                data-track-click="vod-related-clips"
                data-test-selector={TestSelectors.ClipsOverviewURL}
                targetBlank
                linkTo={vod.clipsOverviewURL}
                disabled={!vod.clipsOverviewURL}
                type={ButtonType.Hollow}
              >
                Related Clips
              </Button>
            </Tooltip>

            <Layout display={Display.Flex} justifyContent={JustifyContent.Center}>
              <DeleteVodAndClipsButton
                id={vod.id}
                forceDisabledReason={this.shouldShowUnmutedEmbed ? "Internal vods cannot be deleted" : undefined}
              />
            </Layout>
          </Layout>
        </Layout>
      </Layout>
    );
  }

  @action
  private updateEmbed: React.FormEventHandler<HTMLInputElement> = e => {
    const value = (e.target as HTMLInputElement).value;
    switch (value) {
      case DisplayOption.Twitch:
        this.videoOption = DisplayOption.Twitch;
        break;
      case DisplayOption.Unmuted:
        this.videoOption = DisplayOption.Unmuted;
        break;
      default:
        console.error("Unknown display option", value);
    }
  };

  @computed
  get shouldShowTwitchEmbed() {
    return this.videoOption === DisplayOption.Twitch;
  }

  @computed
  get shouldShowUnmutedEmbed() {
    return this.videoOption === DisplayOption.Unmuted;
  }
}
