import * as React from "react";

import * as Enzyme from "enzyme";
import * as Adapter from "enzyme-adapter-react-16";

import { ConfirmationModalConfirmSelector } from "aegis/functionality/components/confirmation-modal/component";
import { mockErrorResponse, mockSuccessResponse } from "aegis/tests/apollo";
import { wrapWithGQLProvider } from "aegis/tests/graphql";
import { waitSomeMs } from "aegis/tests/utils/wait-some-ms";
import { Button } from "twitch-core-ui";
import { DELETE_VOD_AND_CLIPS, DeleteVodAndClipsButton, Props } from ".";
import { TestSelectors } from "./component";

Enzyme.configure({ adapter: new Adapter() });

const successInput = {
  id: "success"
};

const errorInput = {
  id: "error"
};

const mockDeletionSuccess = mockSuccessResponse(DELETE_VOD_AND_CLIPS, successInput, {
  data: {
    deleteVodAndClips: {
      success: true,
      __typename: "DeleteVodAndClipsPayload"
    }
  }
});

const mockDeletionError = mockErrorResponse(DELETE_VOD_AND_CLIPS, errorInput);

const mocks = [mockDeletionSuccess, mockDeletionError];

function defaultPropsGenerator(overrides?: Partial<Props>): Props {
  return {
    id: "id",
    ...overrides
  };
}

function setupMount(overrides?: Partial<Props>) {
  const props = defaultPropsGenerator();
  return Enzyme.mount(
    wrapWithGQLProvider(<DeleteVodAndClipsButton {...props} {...overrides} />, mocks).wrappedComponent
  );
}

describe("Delete vod and clips button", () => {
  const modalRoot = document.createElement("div");
  modalRoot.setAttribute("id", "portal-modal-root");
  const body = document.querySelector("body");
  if (body) {
    body.appendChild(modalRoot);
  }

  const globalDiv = document.createElement("div");
  document.body.appendChild(globalDiv);

  it("renders", () => {
    const wrapper = setupMount();
    expect(
      wrapper
        .find({ "data-test-selector": TestSelectors.Button })
        .find(Button)
        .prop("disabled")
    ).toBe(false);
  });

  it("disables button when loading gql mutation query", () => {
    const wrapper = setupMount({
      id: successInput.id
    });
    expect(
      wrapper
        .find({ "data-test-selector": TestSelectors.Button })
        .find(Button)
        .prop("disabled")
    ).toBe(false);
    wrapper
      .find({ "data-test-selector": TestSelectors.Button })
      .find(Button)
      .simulate("click");
    wrapper
      .find({ "data-test-selector": ConfirmationModalConfirmSelector })
      .find(Button)
      .simulate("click");
    expect(
      wrapper
        .find({ "data-test-selector": TestSelectors.Button })
        .find(Button)
        .prop("disabled")
    ).toBe(true);
  });

  it("disables button if message is successfully deleted", async () => {
    const wrapper = setupMount({
      id: successInput.id
    });
    expect(
      wrapper
        .find({ "data-test-selector": TestSelectors.Button })
        .find(Button)
        .prop("disabled")
    ).toBe(false);
    wrapper
      .find({ "data-test-selector": TestSelectors.Button })
      .find(Button)
      .simulate("click");
    wrapper
      .find({ "data-test-selector": ConfirmationModalConfirmSelector })
      .find(Button)
      .simulate("click");
    await waitSomeMs();
    wrapper.update();
    expect(
      wrapper
        .find({ "data-test-selector": TestSelectors.Button })
        .find(Button)
        .prop("disabled")
    ).toBe(true);
  });

  it("does not disables button if deletion is unsuccessful", async () => {
    spyOn(console, "error"); // console error is expected

    const wrapper = setupMount({
      id: errorInput.id
    });
    expect(
      wrapper
        .find({ "data-test-selector": TestSelectors.Button })
        .find(Button)
        .prop("disabled")
    ).toBe(false);
    wrapper
      .find({ "data-test-selector": TestSelectors.Button })
      .find(Button)
      .simulate("click");
    wrapper
      .find({ "data-test-selector": ConfirmationModalConfirmSelector })
      .find(Button)
      .simulate("click");
    await waitSomeMs();
    wrapper.update();
    expect(
      wrapper
        .find({ "data-test-selector": TestSelectors.Button })
        .find(Button)
        .prop("disabled")
    ).toBe(false);
  });
});
