import * as React from "react";

import { OperationDefinitionNode } from "graphql";
import ErrorBoundary from "react-error-boundary";

import { gqlStorageKey } from "aegis/models/gql_request";
import { OperationVariables, Query as ApolloQuery, QueryProps as ApolloQueryProps } from "react-apollo";
import { Color, CoreText } from "twitch-core-ui";
import { RollbarHandler } from "../rollbar";

export interface QueryProps {
  displayName: string;
  fallbackComponent?: React.ReactNode;
  context?: {};
}

interface GQLContext {
  operationName: string;
  lastRequestID: string | null;
}

export class Query<Data = {}, Variables = OperationVariables> extends React.Component<
  QueryProps & ApolloQueryProps<Data, Variables>
> {
  public render() {
    const props = this.props;
    const { displayName, context, fallbackComponent } = this.props;

    return (
      <ErrorBoundary
        FallbackComponent={({ error, componentStack }) => (
          <RollbarHandler
            componentName={displayName}
            context={{
              graphqlContext: this.getGqlContexts(),
              ...context
            }}
            error={error}
            componentStack={componentStack}
          >
            <CoreText color={Color.Error}>
              An unexpected error occurred. The Safety Dev team has been notified, but please notify your Shift Leader
              to page us if this is severely impacting your work.
            </CoreText>
            {fallbackComponent}
          </RollbarHandler>
        )}
      >
        {/* work around typing issue https://github.com/DefinitelyTyped/DefinitelyTyped/issues/32588 */}
        <ApolloQuery {...this.props as JSX.LibraryManagedAttributes<typeof ApolloQuery, typeof props>}>
          {result => {
            return this.props.children(result);
          }}
        </ApolloQuery>
      </ErrorBoundary>
    );
  }

  getGqlContexts = () => {
    const { query } = this.props;

    let gqlContexts: GQLContext[] = [];
    if (query) {
      query.definitions.forEach(q => {
        if (q.kind === "OperationDefinition") {
          const op = q as OperationDefinitionNode;

          gqlContexts = gqlContexts.concat({
            operationName: op.name ? op.name.value : "unnamed",
            lastRequestID: op.name ? sessionStorage.getItem(gqlStorageKey(op.name.value)) : "unknown"
          });
        }
      });
    }
    return gqlContexts;
  };
}
